/*
 * Copyright (c) 2011-2015, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2014 KYOCERA Corporation
 * (C) 2016 KYOCERA Corporation
 * (C) 2019 KYOCERA Corporation
 */

#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/netdevice.h>
#include <linux/spinlock.h>
#include <linux/usb_bam.h>
#include <linux/timer.h>

#include "usb_gadget_xport.h"
#include "u_ether.h"
#include "u_rmnet.h"
#include "gadget_chips.h"

#include "u_rmnet_ctrl_tty.c"
#include "kcusb_log.h"

#define TGET_TIME	(jiffies + msecs_to_jiffies(10000))

#define USB_LOG_SIZE 192 /* 64byte per line */
#define USB_LOG_SEND 0
#define USB_LOG_RECV 1

#define USB_LOG_BINARY_A 0x0A
#define USB_LOG_ASCII_0  0x30
#define USB_LOG_ASCII_A  0x37
#define USB_LOG_ASCII_C  0x2C

static int fdm_usb_dbg_mode = 0;

static bool	tx_dm_pend_flg = false;
static bool	dm_suspend = false;

/* TODO: use separate structures for data and
 * control paths
 */
struct f_dm {
	struct gether			gether_port;
	struct grmnet			port;
	int				ifc_id;
	atomic_t			online;
	atomic_t			ctrl_online;
	struct usb_composite_dev	*cdev;

	spinlock_t			lock;

	/* usb eps*/
	struct usb_ep			*notify;
	struct usb_request		*notify_req;

	/* control info */
	struct list_head		cpkt_resp_q;
	unsigned long			notify_count;
	unsigned long			cpkts_len;
	struct delayed_work		work;

	char logbuf[USB_LOG_SIZE + 1];
};

static struct dm_ports {
	struct f_dm			*port;
	struct timer_list		timer;
} dm_ports;

static struct usb_interface_descriptor dm_interface_desc = {
	.bLength =				USB_DT_INTERFACE_SIZE,
	.bDescriptorType =		USB_DT_INTERFACE,
	/* .bInterfaceNumber = 	DYNAMIC */
	.bNumEndpoints =		1,
	.bInterfaceClass =		USB_CLASS_COMM,
	.bInterfaceSubClass =	USB_CDC_SUBCLASS_DMM,
	.bInterfaceProtocol =	USB_CDC_ACM_PROTO_AT_V25TER,
	/* .iInterface = DYNAMIC */
};

static struct usb_cdc_header_desc dm_header_desc = {
	.bLength =				sizeof(dm_header_desc),
	.bDescriptorType =		USB_DT_CS_INTERFACE,
	.bDescriptorSubType =	USB_CDC_HEADER_TYPE,
	.bcdCDC =				cpu_to_le16(0x0110),
};

static struct usb_cdc_dmm_desc dm_descriptor = {
	.bFunctionLength =		sizeof(dm_descriptor),
	.bDescriptorType =		USB_DT_CS_INTERFACE,
	.bDescriptorSubtype =	USB_CDC_DMM_TYPE,
	.bcdVersion =			cpu_to_le16(0x0100),
	.wMaxCommand =			cpu_to_le16(0x0100),
};

/* Full speed support */
static struct usb_endpoint_descriptor dm_fs_notify_desc = {
	.bLength =			USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_IN,
	.bmAttributes =		USB_ENDPOINT_XFER_INT,
	.wMaxPacketSize =	cpu_to_le16(0x10),
	.bInterval =		0x10,
};

static struct usb_descriptor_header *dm_fs_function[] = {
	(struct usb_descriptor_header *) &dm_interface_desc,
	(struct usb_descriptor_header *) &dm_header_desc,
	(struct usb_descriptor_header *) &dm_descriptor,
	(struct usb_descriptor_header *) &dm_fs_notify_desc,
	NULL,
};

/* High speed support */
static struct usb_endpoint_descriptor dm_hs_notify_desc = {
	.bLength =			USB_DT_ENDPOINT_SIZE,
	.bDescriptorType =	USB_DT_ENDPOINT,
	.bEndpointAddress =	USB_DIR_IN,
	.bmAttributes =		USB_ENDPOINT_XFER_INT,
	.wMaxPacketSize =	cpu_to_le16(0x10),
	.bInterval =		0x06,
};

static struct usb_descriptor_header *dm_hs_function[] = {
	(struct usb_descriptor_header *) &dm_interface_desc,
	(struct usb_descriptor_header *) &dm_header_desc,
	(struct usb_descriptor_header *) &dm_descriptor,
	(struct usb_descriptor_header *) &dm_hs_notify_desc,
	NULL,
};

/* String descriptors */

static struct usb_string dm_string_defs[] = {
	[0].s = "Device Management Communication Class Interface",
	{  } /* end of list */
};

static struct usb_gadget_strings dm_string_table = {
	.language =		0x0409,	/* en-us */
	.strings =		dm_string_defs,
};

static struct usb_gadget_strings *dm_strings[] = {
	&dm_string_table,
	NULL,
};

static void fdm_ctrl_response_available(struct f_dm *dev);
static int fdm_init_port(void);

bool tget_null_packet = false;

/*
 * fdm_set_usb_dbg_mode
 */
void fdm_set_usb_dbg_mode (int mode)
{
	fdm_usb_dbg_mode = mode;
}
EXPORT_SYMBOL_GPL(fdm_set_usb_dbg_mode);

/*
 * fdm_usb_data_log
 *
 * Called when liboem_usb_port_write or liboem_usb_port_read is executed.
 * Outputs the transmitted / received data to logcat.
 *
 */
static void fdm_usb_data_log (struct f_dm *port, int type, int size, char *buff)
{
	int pos = 0;
	int i   = 0;
	
	if (fdm_usb_dbg_mode != 1) {
		return;
	}
	
	if ((!port) || (!buff) || (size <= 0)) {
		return;
	}

	if (type == USB_LOG_SEND) {
		pr_info("Send : data size=%d(byte)\n", size);
	} else if (type == USB_LOG_RECV) {
		pr_info("Recv : data size=%d(byte)\n", size);
	}

	memset(&port->logbuf[0], 0, (USB_LOG_SIZE + 1));	
	for (i = 0; i < size; i++) {
		port->logbuf[pos++] = (((buff[i] & 0xF0) >> 4) < USB_LOG_BINARY_A) ?
			(((buff[i] & 0xF0) >> 4) + USB_LOG_ASCII_0) :
			(((buff[i] & 0xF0) >> 4) + USB_LOG_ASCII_A);
		port->logbuf[pos++] = (((buff[i] & 0x0F) >> 0) < USB_LOG_BINARY_A) ?
			(((buff[i] & 0x0F) >> 0) + USB_LOG_ASCII_0) :
			(((buff[i] & 0x0F) >> 0) + USB_LOG_ASCII_A);
		port->logbuf[pos++] = USB_LOG_ASCII_C;
		port->logbuf[pos] = 0x00;
		
		if (USB_LOG_SIZE <= pos) {
			pr_info("%s\n", &port->logbuf[0]);
			pos = 0;
		}
	}
	
	if (pos) {
		pr_info("%s\n", &port->logbuf[0]);
	}
}

/* ------- misc functions --------------------*/

static inline struct f_dm *func_to_dm(struct usb_function *f)
{
	return container_of(f, struct f_dm, gether_port.func);
}

static inline struct f_dm *port_to_dm(struct grmnet *r)
{
	return container_of(r, struct f_dm, port);
}

static struct usb_request *
fdm_alloc_req(struct usb_ep *ep, unsigned len, size_t extra_buf_alloc,
		gfp_t flags)
{
	struct usb_request *req;

	req = usb_ep_alloc_request(ep, flags);
	if (!req)
		return ERR_PTR(-ENOMEM);

	req->buf = kmalloc(len + extra_buf_alloc, flags);
	if (!req->buf) {
		usb_ep_free_request(ep, req);
		return ERR_PTR(-ENOMEM);
	}

	req->length = len;

	USBLOGOUT("DM: %s\n", ep->address & USB_ENDPOINT_DIR_MASK ? "tba" : "rba");

	return req;
}

void fdm_free_req(struct usb_ep *ep, struct usb_request *req)
{
	USBLOGOUT("DM: %s\n", ep->address & USB_ENDPOINT_DIR_MASK ? "tbf" : "rbf");
	kfree(req->buf);
	usb_ep_free_request(ep, req);
}

static struct rmnet_ctrl_pkt *dm_alloc_ctrl_pkt(unsigned len, gfp_t flags)
{
	struct rmnet_ctrl_pkt *pkt;

	pkt = kzalloc(sizeof(struct rmnet_ctrl_pkt), flags);
	if (!pkt)
		return ERR_PTR(-ENOMEM);

	pkt->buf = kmalloc(len, flags);
	if (!pkt->buf) {
		kfree(pkt);
		return ERR_PTR(-ENOMEM);
	}
	pkt->len = len;

	return pkt;
}

static void dm_free_ctrl_pkt(struct rmnet_ctrl_pkt *pkt)
{
	kfree(pkt->buf);
	kfree(pkt);
}

/* -------------------------------------------*/

static int gport_dm_connect(struct f_dm *dev, unsigned intf)
{
	int			ret;
	unsigned		port_num;

	pr_debug("%s: ctrl xport: %s data xport: %s dev: %p portno: %d\n",
			__func__, "tty", "tty",
			dev, 0);

	port_num = 0;
	ret = gtty_ctrl_connect(&dev->port, port_num);
	if (ret) {
		pr_err("%s: gtty_ctrl_connect failed: err:%d\n",
				__func__, ret);
		return ret;
	}

	return 0;
}

static int gport_dm_disconnect(struct f_dm *dev)
{
	unsigned		port_num;

	pr_debug("%s: ctrl xport: %s data xport: %s dev: %p portno: %d\n",
			__func__, "tty", "tty",
			dev, 0);

	port_num = 0;
	gtty_ctrl_disconnect(&dev->port, port_num);

	return 0;
}

static void fdm_unbind(struct usb_configuration *c, struct usb_function *f)
{
	struct f_dm *dev = func_to_dm(f);

	pr_debug("%s: portno:%d\n", __func__, 0);
	if (gadget_is_dualspeed(c->cdev->gadget))
		usb_free_descriptors(f->hs_descriptors);
	usb_free_descriptors(f->fs_descriptors);

	fdm_free_req(dev->notify, dev->notify_req);
}

static void fdm_purge_responses(struct f_dm *dev)
{
	unsigned long flags;
	struct rmnet_ctrl_pkt *cpkt;

	pr_debug("%s: port#%d\n", __func__, 0);

	spin_lock_irqsave(&dev->lock, flags);
	while (!list_empty(&dev->cpkt_resp_q)) {
		cpkt = list_first_entry(&dev->cpkt_resp_q,
				struct rmnet_ctrl_pkt, list);

		list_del(&cpkt->list);
		USBLOGOUT("DM: tbc%d\n", cpkt->len);
		dm_free_ctrl_pkt(cpkt);
	}
	dm_suspend = false;
	tx_dm_pend_flg = false;

	dev->notify_count = 0;
	spin_unlock_irqrestore(&dev->lock, flags);

	pr_debug("%s dm_ports.timer del_timer\n", __func__);
	del_timer_sync(&dm_ports.timer);
}

static void fdm_suspend(struct usb_function *f)
{
	struct f_dm *dev = func_to_dm(f);
	pr_debug("%s: data xport: %s dev: %p portno: %d remote_wakeup: %d\n",
		__func__, "tty", dev, 0, 0);

	dm_suspend = true;
}

static void fdm_resume(struct usb_function *f)
{
	unsigned long flags;
	struct f_dm *dev = func_to_dm(f);
	pr_debug("%s: data xport: %s dev: %p portno: %d remote_wakeup: %d\n",
		__func__, "tty", dev, 0, 0);

	dm_suspend = false;
	spin_lock_irqsave(&dev->lock, flags);
	if (!dev->notify_count) {
		tx_dm_pend_flg = false;
	} else {
		if (tx_dm_pend_flg == true) {
			pr_debug("func:%s dev->work", __func__);
			schedule_delayed_work(&dev->work, msecs_to_jiffies(0));
		}
	}
	spin_unlock_irqrestore(&dev->lock, flags);
}

static void fdm_disable(struct usb_function *f)
{
	struct f_dm *dev = func_to_dm(f);

	pr_debug("%s: port#%d\n", __func__, 0);

	usb_ep_disable(dev->notify);
	dev->notify->driver_data = NULL;

	atomic_set(&dev->online, 0);

	fdm_purge_responses(dev);
	gport_dm_disconnect(dev);
}

static int
fdm_set_alt(struct usb_function *f, unsigned intf, unsigned alt)
{
	struct f_dm			*dev = func_to_dm(f);
	struct usb_composite_dev	*cdev = dev->cdev;
	int				ret;
	struct list_head *cpkt;

	pr_debug("%s:dev:%p port#%d\n", __func__, dev, 0);

	if (dev->notify->driver_data) {
		pr_debug("%s: reset port:%d\n", __func__, 0);
		usb_ep_disable(dev->notify);
	}

	ret = config_ep_by_speed(cdev->gadget, f, dev->notify);
	if (ret) {
		dev->notify->desc = NULL;
		pr_err("config_ep_by_speed failes for ep %s, result %d\n",
					dev->notify->name, ret);
		return ret;
	}
	ret = usb_ep_enable(dev->notify);

	if (ret) {
		pr_err("%s: usb ep#%s enable failed, err#%d\n",
				__func__, dev->notify->name, ret);
		dev->notify->desc = NULL;
		return ret;
	}
	dev->notify->driver_data = dev;

	ret = gport_dm_connect(dev, intf);
	if (ret) {
		pr_err("%s(): gport_dm_connect fail with err:%d\n",
							__func__, ret);
		goto err_unconfig_ep;
	}

	atomic_set(&dev->online, 1);

	/* In case notifications were aborted, but there are pending control
	   packets in the response queue, re-add the notifications */
	list_for_each(cpkt, &dev->cpkt_resp_q)
		fdm_ctrl_response_available(dev);

	return ret;

err_unconfig_ep:
	usb_ep_disable(dev->notify);

	return ret;
}

static void fdm_ctrl_response_available(struct f_dm *dev)
{
	struct usb_request		*req = dev->notify_req;
	struct usb_cdc_notification	*event;
	unsigned long			flags;
	int				ret;
	struct rmnet_ctrl_pkt	*cpkt;

	pr_debug("%s:dev:%p portno#%d\n", __func__, dev, 0);

	spin_lock_irqsave(&dev->lock, flags);
	if (!atomic_read(&dev->online) || !req || !req->buf) {
		spin_unlock_irqrestore(&dev->lock, flags);
		return;
	}

	if (++dev->notify_count != 1) {
		spin_unlock_irqrestore(&dev->lock, flags);
		return;
	}

	event = req->buf;
	event->bmRequestType = USB_DIR_IN | USB_TYPE_CLASS
			| USB_RECIP_INTERFACE;
	event->bNotificationType = USB_CDC_NOTIFY_RESPONSE_AVAILABLE;
	event->wValue = cpu_to_le16(0);
	event->wIndex = cpu_to_le16(dev->ifc_id);
	event->wLength = cpu_to_le16(0);
	spin_unlock_irqrestore(&dev->lock, flags);

	ret = usb_ep_queue(dev->notify, dev->notify_req, GFP_ATOMIC);
	if (ret) {
		if (dm_suspend == true) {
			pr_err("%s: ep_queue error suspend\n", __func__);
			tx_dm_pend_flg = true;
			return;
		}

		spin_lock_irqsave(&dev->lock, flags);
		if (!list_empty(&dev->cpkt_resp_q)) {
			if (dev->notify_count > 0)
				dev->notify_count--;
			else {
				pr_debug("%s: Invalid notify_count=%lu to decrement\n",
					 __func__, dev->notify_count);
				spin_unlock_irqrestore(&dev->lock, flags);
				return;
			}
			cpkt = list_first_entry(&dev->cpkt_resp_q,
					struct rmnet_ctrl_pkt, list);
			list_del(&cpkt->list);
			USBLOGOUT("DM: tbc%d\n", cpkt->len);
			dm_free_ctrl_pkt(cpkt);
		}
		spin_unlock_irqrestore(&dev->lock, flags);
		pr_debug("ep enqueue error %d\n", ret);
	} else {
		USBLOGOUT("DM: un\n");
	}
}

static void fdm_connect(struct grmnet *gr)
{
	struct f_dm			*dev;

	if (!gr) {
		pr_err("%s: Invalid grmnet:%p\n", __func__, gr);
		return;
	}

	dev = port_to_dm(gr);

	atomic_set(&dev->ctrl_online, 1);
}

static void fdm_disconnect(struct grmnet *gr)
{
	struct f_dm			*dev;

	if (!gr) {
		pr_err("%s: Invalid grmnet:%p\n", __func__, gr);
		return;
	}

	dev = port_to_dm(gr);

	atomic_set(&dev->ctrl_online, 0);

	if (!atomic_read(&dev->online)) {
		pr_debug("%s: nothing to do\n", __func__);
		return;
	}

	fdm_purge_responses(dev);
}

static int
fdm_send_cpkt_response(void *gr, void *buf, size_t len)
{
	struct f_dm		*dev;
	struct rmnet_ctrl_pkt	*cpkt;
	unsigned long		flags;

	if (!gr || !buf) {
		pr_err("%s: Invalid grmnet/buf, grmnet:%p buf:%p\n",
				__func__, gr, buf);
		return -ENODEV;
	}
	cpkt = dm_alloc_ctrl_pkt(len, GFP_ATOMIC);
	if (IS_ERR(cpkt)) {
		pr_err("%s: Unable to allocate ctrl pkt\n", __func__);
		return -ENOMEM;
	}
	memcpy(cpkt->buf, buf, len);
	cpkt->len = len;

	dev = port_to_dm(gr);

	pr_debug("%s: dev:%p port#%d len=%d\n", __func__, dev, 0, len);

	if (!atomic_read(&dev->online) || !atomic_read(&dev->ctrl_online)) {
		USBLOGOUT("DM: tbc%d\n", cpkt->len);
		dm_free_ctrl_pkt(cpkt);
		return 0;
	}

	spin_lock_irqsave(&dev->lock, flags);
	list_add_tail(&cpkt->list, &dev->cpkt_resp_q);
	spin_unlock_irqrestore(&dev->lock, flags);

	fdm_ctrl_response_available(dev);

	return 0;
}

static void
fdm_cmd_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct f_dm			*dev = req->context;
	struct usb_composite_dev	*cdev;
	unsigned			port_num;

	USBLOGOUT("DM: ur%d\n", req->length);
	USBLOGOUT("DM: rc%d\n", req->status);

	if (!dev) {
		pr_err("%s: dm dev is null\n", __func__);
		return;
	}

	pr_debug("%s: dev:%p port#%d\n", __func__, dev, 0);

	cdev = dev->cdev;

	if (dev->port.send_encap_cmd) {
		port_num = 0;
		fdm_usb_data_log(dev, USB_LOG_RECV, req->actual, req->buf);
		dev->port.send_encap_cmd(port_num, req->buf, req->actual);
	}
}

static void
fdm_get_encap_response_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct f_dm		*dev = req->context;
	unsigned long	flags;

	USBLOGOUT("DM: tc%d\n", req->status);

	if (!dev) {
		pr_err("%s: dm dev is null\n", __func__);
		return;
	}

	pr_debug("%s: dev:%p\n", __func__, dev);

	if (!atomic_read(&dev->ctrl_online))
		return;

	spin_lock_irqsave(&dev->lock, flags);
	if (dev->notify_count == 0) {
		spin_unlock_irqrestore(&dev->lock, flags);
		return;
	}

	if (--dev->notify_count == 0) {
		spin_unlock_irqrestore(&dev->lock, flags);
		return;
	}
	spin_unlock_irqrestore(&dev->lock, flags);
	schedule_delayed_work(&dev->work, msecs_to_jiffies(0));
}

#define NOTIFY_RETRY	5
static void fdm_notify_work(struct work_struct *data)
{
	struct f_dm			*dev = container_of(data, struct f_dm, work.work);
	int					status;
	unsigned long		flags;
	struct rmnet_ctrl_pkt	*cpkt;
	static int			retry = NOTIFY_RETRY;

	pr_debug("%s\n", __func__);
	if (!atomic_read(&dev->online)) {
		pr_debug("%s: nothing to do\n", __func__);
		return;
	}
	status = usb_ep_queue(dev->notify, dev->notify_req, GFP_ATOMIC);
	if (status) {
		if (dm_suspend == true) {
			pr_err("%s: ep_queue error suspend\n", __func__);
			tx_dm_pend_flg = true;
			return;
		}
		if (!retry) {
			spin_lock_irqsave(&dev->lock, flags);
			if (dev->notify_count > 0)
				dev->notify_count--;
			if (!list_empty(&dev->cpkt_resp_q)) {
				cpkt = list_first_entry(&dev->cpkt_resp_q,
					struct rmnet_ctrl_pkt, list);
				if (cpkt) {
					list_del(&cpkt->list);
					USBLOGOUT("DM: tbc%d\n", cpkt->len);
					dm_free_ctrl_pkt(cpkt);
				}
			}
			spin_unlock_irqrestore(&dev->lock, flags);
			pr_debug("ep enqueue error %d\n", status);
			retry = NOTIFY_RETRY;
		} else {
			--retry;
			schedule_delayed_work(&dev->work, msecs_to_jiffies(1));
		}
	} else {
		tx_dm_pend_flg = false;
		retry = NOTIFY_RETRY;
	}
}

static void fdm_notify_complete(struct usb_ep *ep, struct usb_request *req)
{
	struct f_dm *dev = req->context;
	int status = req->status;
	unsigned long		flags;

	pr_debug("%s: dev:%p port#%d\n", __func__, dev, 0);
	USBLOGOUT("DM: nc%d\n", req->status);

	switch (status) {
	case -ECONNRESET:
	case -ESHUTDOWN:
		/* connection gone */
		spin_lock_irqsave(&dev->lock, flags);
		dev->notify_count = 0;
		spin_unlock_irqrestore(&dev->lock, flags);
		break;
	default:
		pr_err("dm notify ep error %d\n", status);
		/* FALLTHROUGH */
	case 0:
		spin_lock_irqsave(&dev->lock, flags);
		if (dev->notify_count) {
			pr_debug("%s dm_ports.timer mod_timer\n", __func__);
			mod_timer(&dm_ports.timer, TGET_TIME);
		}
		spin_unlock_irqrestore(&dev->lock, flags);
		break;
	}
}

static int
fdm_setup(struct usb_function *f, const struct usb_ctrlrequest *ctrl)
{
	struct f_dm			*dev = func_to_dm(f);
	struct usb_composite_dev	*cdev = dev->cdev;
	struct usb_request		*req = cdev->req;
	u16				w_index = le16_to_cpu(ctrl->wIndex);
	u16				w_value = le16_to_cpu(ctrl->wValue);
	u16				w_length = le16_to_cpu(ctrl->wLength);
	int				ret = -EOPNOTSUPP;

	pr_debug("%s:dev:%p port#%d\n", __func__, dev, 0);

	if (!atomic_read(&dev->online)) {
		pr_warning("%s: usb cable is not connected\n", __func__);
		return -ENOTCONN;
	}

	switch ((ctrl->bRequestType << 8) | ctrl->bRequest) {

	case ((USB_DIR_OUT | USB_TYPE_CLASS | USB_RECIP_INTERFACE) << 8)
			| USB_CDC_SEND_ENCAPSULATED_COMMAND:
		pr_debug("%s: USB_CDC_SEND_ENCAPSULATED_COMMAND\n"
				 , __func__);
		if (w_value)
			goto invalid;

		ret = w_length;
		req->complete = fdm_cmd_complete;
		req->context = dev;
		break;


	case ((USB_DIR_IN | USB_TYPE_CLASS | USB_RECIP_INTERFACE) << 8)
			| USB_CDC_GET_ENCAPSULATED_RESPONSE:
		pr_debug("%s: USB_CDC_GET_ENCAPSULATED_RESPONSE\n", __func__);
		if (w_value) {
			pr_err("%s: invalid w_value = %04x\n",
				   __func__ , w_value);
			goto invalid;
		} else {
			unsigned len;
			struct rmnet_ctrl_pkt *cpkt;

			pr_debug("%s dm_ports.timer del_timer\n", __func__);
			del_timer_sync(&dm_ports.timer);

			req->complete = fdm_get_encap_response_complete;
			req->context = dev;

			spin_lock(&dev->lock);
			if (list_empty(&dev->cpkt_resp_q)) {
				ret = 0;
				spin_unlock(&dev->lock);
				tget_null_packet = true;
				pr_debug("transfer null packet\n");
				goto transfer;
			}

			cpkt = list_first_entry(&dev->cpkt_resp_q,
					struct rmnet_ctrl_pkt, list);
			len = min_t(unsigned, w_length, cpkt->len);
			memcpy(req->buf, cpkt->buf, len);
			ret = len;

			if (w_length < cpkt->len) {
				cpkt->len -= w_length;
				memmove(cpkt->buf, cpkt->buf + w_length, cpkt->len);
				dev->notify_count++;
			} else {
				list_del(&cpkt->list);
				dm_free_ctrl_pkt(cpkt);
			}
			USBLOGOUT("DM: ut%d\n", len);
			spin_unlock(&dev->lock);
			
			fdm_usb_data_log(dev, USB_LOG_SEND, len, req->buf);
		}
		break;
	default:

invalid:
		pr_debug("invalid control req%02x.%02x v%04x i%04x l%d\n",
			ctrl->bRequestType, ctrl->bRequest,
			w_value, w_index, w_length);
	}

transfer:
	/* respond with data transfer or status phase? */
	if (ret >= 0) {
		pr_debug("dm req%02x.%02x v%04x i%04x l%d\n",
			ctrl->bRequestType, ctrl->bRequest,
			w_value, w_index, w_length);
		req->zero = (ret < w_length);
		req->length = ret;
		ret = usb_ep_queue(cdev->gadget->ep0, req, GFP_ATOMIC);
		if (ret < 0)
			pr_err("dm ep0 enqueue err %d\n", ret);
	}

	return ret;
}

static int fdm_bind(struct usb_configuration *c, struct usb_function *f)
{
	struct f_dm			*dev = func_to_dm(f);
	struct usb_ep			*ep;
	struct usb_composite_dev	*cdev = c->cdev;
	int				ret = -ENODEV;
	pr_debug("%s: start binding\n", __func__);
	dev->ifc_id = usb_interface_id(c, f);
	if (dev->ifc_id < 0) {
		pr_err("%s: unable to allocate ifc id, err:%d\n",
				__func__, dev->ifc_id);
		return dev->ifc_id;
	}
	dm_interface_desc.bInterfaceNumber = dev->ifc_id;

	ep = usb_ep_autoconfig(cdev->gadget, &dm_fs_notify_desc);
	if (!ep) {
		pr_err("%s: usb epnotify autoconfig failed\n", __func__);
		ret = -ENODEV;
		goto ep_auto_notify_fail;
	}
	dev->notify = ep;
	ep->driver_data = cdev;

	dev->notify_req = fdm_alloc_req(ep,
				sizeof(struct usb_cdc_notification),
				cdev->gadget->extra_buf_alloc,
				GFP_KERNEL);
	if (IS_ERR(dev->notify_req)) {
		pr_err("%s: unable to allocate memory for notify req\n",
				__func__);
		ret = -ENOMEM;
		goto ep_notify_alloc_fail;
	}

	dev->notify_req->complete = fdm_notify_complete;
	dev->notify_req->context = dev;

	ret = -ENOMEM;
	f->fs_descriptors = usb_copy_descriptors(dm_fs_function);

	if (!f->fs_descriptors) {
		pr_err("%s: no descriptors,usb_copy descriptors(fs)failed\n",
			__func__);
		goto fail;
	}
	if (gadget_is_dualspeed(cdev->gadget)) {
		dm_hs_notify_desc.bEndpointAddress =
				dm_fs_notify_desc.bEndpointAddress;

		/* copy descriptors, and track endpoint copies */
		f->hs_descriptors = usb_copy_descriptors(dm_hs_function);

		if (!f->hs_descriptors) {
			pr_err("%s: no hs_descriptors,usb_copy descriptors(hs)failed\n",
			__func__);
			goto fail;
		}
	}


	pr_debug("%s: RmNet(%d) %s Speed, IN:%s OUT:%s\n",
			__func__, 0,
			gadget_is_dualspeed(cdev->gadget) ? "dual" : "full",
			dev->port.in->name, dev->port.out->name);

	return 0;

fail:
	if (f->hs_descriptors)
		usb_free_descriptors(f->hs_descriptors);
	if (f->fs_descriptors)
		usb_free_descriptors(f->fs_descriptors);
	if (dev->notify_req)
		fdm_free_req(dev->notify, dev->notify_req);
ep_notify_alloc_fail:
	dev->notify->driver_data = NULL;
	dev->notify = NULL;
ep_auto_notify_fail:

	return ret;
}

static int fdm_bind_config(struct usb_configuration *c)
{
	int			status;
	struct f_dm		*dev;
	struct usb_function	*f;
	unsigned long		flags;
	static int			dm_initialized = 0;

	pr_debug("%s: usb config:%p\n", __func__, c);

	/* String Descriptor always rebuilds. */
	dm_string_defs[0].id = 0;

	if (!dm_initialized) {
		dm_initialized = 1;
		status = fdm_init_port();
		if (status)
			return status;

		status = gtty_ctrl_setup(FRMNET_CTRL_CLIENT, 1, 0);
		if (status)
			return status;
	}

	dev = dm_ports.port;

	if (dm_string_defs[0].id == 0) {
		status = usb_string_id(c->cdev);
		if (status < 0) {
			pr_err("%s: failed to get string id, err:%d\n",
					__func__, status);
			return status;
		}
		dm_string_defs[0].id = status;
		dm_interface_desc.iInterface = status;
	}

	spin_lock_irqsave(&dev->lock, flags);
	dev->cdev = c->cdev;
	f = &dev->gether_port.func;
	spin_unlock_irqrestore(&dev->lock, flags);

	f->name = "dm";
	f->strings = dm_strings;
	f->bind = fdm_bind;
	f->unbind = fdm_unbind;
	f->disable = fdm_disable;
	f->set_alt = fdm_set_alt;
	f->setup = fdm_setup;
	f->suspend = fdm_suspend;
	f->resume = fdm_resume;
	dev->port.send_cpkt_response = fdm_send_cpkt_response;
	dev->port.disconnect = fdm_disconnect;
	dev->port.connect = fdm_connect;
	dev->gether_port.cdc_filter = 0;

	status = usb_add_function(c, f);
	if (status) {
		pr_err("%s: usb add function failed: %d\n",
				__func__, status);
		return status;
	}

	pr_debug("%s: complete\n", __func__);

	return status;
}

static void fdm_unbind_config(void)
{
}

static int dm_init(void)
{
	return 0;
}

static void fdm_cleanup(void)
{
	cancel_delayed_work_sync(&dm_ports.port->work);
	gserial_free_line_tty(0);
	kfree(dm_ports.port);
}

static void fdm_timer_func(unsigned long data)
{
	struct f_dm			*dev = dm_ports.port;
	unsigned long		flags;
	struct rmnet_ctrl_pkt	*cpkt;

	pr_debug("%s called.\n", __func__);
	USBLOGOUT("DM: ubc\n");

	spin_lock_irqsave(&dev->lock, flags);
	while (!list_empty(&dev->cpkt_resp_q)) {
		cpkt = list_first_entry(&dev->cpkt_resp_q,
			struct rmnet_ctrl_pkt, list);

		pr_debug("%s dm_free_ctrl_pkt len=%d\n", __func__, cpkt->len);
		list_del(&cpkt->list);
		USBLOGOUT("DM: tbc%d\n", cpkt->len);
		dm_free_ctrl_pkt(cpkt);
	}
	gtty_ctrl_read_pending(0);
	dev->notify_count = 0;
	spin_unlock_irqrestore(&dev->lock, flags);
}

static int fdm_init_port(void)
{
	struct f_dm			*dev;

	pr_debug("%s: port#:%d, ctrl port: %s data port: %s\n",
		__func__, 0, "-", "-");

	dev = kzalloc(sizeof(struct f_dm), GFP_KERNEL);
	if (!dev) {
		pr_err("%s: Unable to allocate dm device\n", __func__);
		return -ENOMEM;
	}

	spin_lock_init(&dev->lock);
	INIT_LIST_HEAD(&dev->cpkt_resp_q);
	INIT_DELAYED_WORK(&dev->work, fdm_notify_work);
	init_timer(&dm_ports.timer);
	dm_ports.timer.function = fdm_timer_func;

	dm_ports.port = dev;

	return 0;
}
