/*
 * pb_smd.c
 *
 * Copyright (c) 2011, The Linux Foundation. All rights reserved.
 *
 * Copyright (C) 2000 - 2003 Al Borchers (alborchers@steinerpoint.com)
 * Copyright (C) 2008 David Brownell
 * Copyright (C) 2008 by Nokia Corporation
 * Copyright (C) 1999 - 2002 Greg Kroah-Hartman (greg@kroah.com)
 * Copyright (C) 2000 Peter Berger (pberger@brimson.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2014 KYOCERA Corporation
 * (C) 2015 KYOCERA Corporation
 * (C) 2018 KYOCERA Corporation
 * (C) 2019 KYOCERA Corporation
 */
#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/slab.h>
#include <linux/termios.h>
#include <soc/qcom/smd.h>
#include <linux/workqueue.h>

#include "pb_command.h"
#include "pb_serial.h"
#include "pb_ring.h"

enum {
    SMD_PORT_DATA4 = 0x00000000,
    SMD_PORT_NO,
};

#define PB_SMD_RING_BUF_SIZE   5120
#define PB_SMD_WRITE_BUF_SIZE  128

extern void android_usb_state_cb_reg (void(*cb)(int a));
static void notify_usb_state (int state);

struct smd_port_info {
	struct smd_channel	*ch;
	const char		*name;
};

struct smd_port_info smd_port_info[SMD_PORT_NO] = {
	{
		.name = "DATA4",
	},
};

struct smd_port_t {
	unsigned		line;
	spinlock_t		port_lock;
	struct mutex		lock;

	int ring;
	char			write_buf[PB_DATA_BUF_SIZE];
	struct work_struct	tx;
	struct delayed_work	smd_tx_retry;

	char			read_buf[PB_DATA_BUF_SIZE];
	struct work_struct	rx;

	struct smd_port_info	*port_info;
	struct delayed_work	connect_work;
};

static struct smd_ports_t {
	struct smd_port_t	*port;
} smd_ports[SMD_PORT_NO];

static struct workqueue_struct	*smd_wq;
static int smd_ready_flag = 0;

static int smd_put_ring(struct smd_port_t *port, char *buf, int len)
{
	int ring = port->ring;
	int retval = 0;
	
	if (pb_ring_is_enough (ring, len)) {
		retval = !pb_ring_push (ring, len, buf);
	} else {
		pb_print(MSG_DEBUG, "Data lost (%d/%d)\n", port->line, len);
		retval = -1;
	}
	
	return retval;
}

static void smd_push_back_ring(struct smd_port_t *port, int len)
{
	int ring = port->ring;

	pb_ring_push_back (ring, len);

}

static int smd_get_ring(struct smd_port_t *port, char *buf, int *len)
{
	int ring = port->ring;
	int retval = 0;
	int getsize = (*len < PB_SMD_WRITE_BUF_SIZE) ? *len : PB_SMD_WRITE_BUF_SIZE;
	
	if (!pb_ring_is_empty (ring)) {
		*len = pb_ring_pull (ring, getsize, buf);
		pb_print (MSG_DEBUG, "Data Found (%d)\n", *len);
	} else {
		pb_print (MSG_DEBUG, "No data in ring (%d)\n", port->line);
		retval = -1;
	}
	
	return retval;
}

static void smd_tx_retry (struct work_struct *work) {
	struct smd_port_t *port =
		container_of(work, struct smd_port_t, smd_tx_retry.work);
	unsigned int ret_busy;
	pb_print(MSG_INFO, "SMD send retry (%d)", port->line);

	ret_busy = work_busy(&port->tx);
	if ( ret_busy == ((unsigned long)(WORK_BUSY_PENDING) | (unsigned long)(WORK_BUSY_RUNNING))){
		pb_print(MSG_DEBUG, "SMD send retry already pending and ruuning\n");
	}else if(ret_busy == (unsigned long)(WORK_BUSY_PENDING)){
		pb_print(MSG_DEBUG, "SMD send retry already pending\n");
	}else if(ret_busy == (unsigned long)(WORK_BUSY_RUNNING)){
		pb_print(MSG_DEBUG, "SMD send retry already running\n");
		queue_delayed_work(smd_wq, &port->smd_tx_retry,msecs_to_jiffies(5));
	}else{
		pb_print(MSG_DEBUG, "SMD send retry no pending and no running\n");
		queue_work(smd_wq, &port->tx);
	}
}
static void smd_tx(struct work_struct *work) {
	struct smd_port_t *port = container_of(work, struct smd_port_t, tx);
	struct smd_port_info *port_info = port->port_info;
	int count;
	int size;
	unsigned long	flags;

	pb_print(MSG_DEBUG, "port:%p port#%d \n", port, port->line);

	spin_lock_irqsave(&port->port_lock, flags);
	if (!port_info->ch) {
		pb_print(MSG_ERROR, "NG ch\n");
		spin_unlock_irqrestore(&port->port_lock, flags);
		return;
	}

	while (1) {
		size = smd_write_avail(port_info->ch);
		if (!size){
			pb_print(MSG_ERROR, "SMD No space to write\n");
			if (work_busy (&port->smd_tx_retry.work) == 0UL) {
				queue_delayed_work(smd_wq, &port->smd_tx_retry,
						msecs_to_jiffies(10));
			}
			break;
		}

		if (smd_get_ring(port, port->write_buf, &size)) {
			pb_print(MSG_DEBUG, "Send end\n");
			break;
		}

		if (size <= 0) {
			pb_print(MSG_ERROR, "Length NG\n");
			break;
		}
		count = smd_write(port_info->ch, port->write_buf, size);
		*(port->write_buf + size) = 0;
		pb_print(MSG_DEBUG, "length = %d data = %s count=%d\n",
			size, port->write_buf, count);
		if (count == size) {
			pb_print(MSG_DEBUG, "Written one\n");
		} else if (count < 0) {
			pb_print(MSG_ERROR, "SMD write error %d - %d\n", port->line, count);
			smd_push_back_ring (port, size);
			queue_delayed_work(smd_wq, &port->smd_tx_retry,
						msecs_to_jiffies(10));
			break;
		} else {
			pb_print(MSG_ERROR, "SMD write busy %d - %d\n", port->line, count);
			smd_push_back_ring (port, size - count);
			queue_delayed_work(smd_wq, &port->smd_tx_retry,
						msecs_to_jiffies(10));
			break;
		}
	}

	spin_unlock_irqrestore(&port->port_lock, flags);

	pb_print(MSG_DEBUG, "Complete\n");

	return;
}

static void smd_notify(void *priv, unsigned event)
{
	struct smd_port_t *port = priv;

	pb_print(MSG_DEBUG, "Entry\n");

	switch (event) {
	case SMD_EVENT_DATA:
		pb_print(MSG_DEBUG, "Event data\n");
		break;
	case SMD_EVENT_OPEN:
		pb_print(MSG_DEBUG, "Event Open port = %d\n", port->line);
		if (port->line == SMD_PORT_DATA4) {
			android_usb_state_cb_reg (notify_usb_state);
			pb_print(MSG_DEBUG, "GPIO check start\n");
		}
		break;
	case SMD_EVENT_CLOSE:
		pb_print(MSG_DEBUG, "Event Close\n");
		break;
	case SMD_EVENT_STATUS:
		pb_print(MSG_DEBUG, "Event status\n");
		break;
	default:
		pb_print(MSG_DEBUG, "Event other%d\n", event);
		break;
	}
}

static void smd_connect_work(struct work_struct *work)
{
	struct smd_port_t *port =
		container_of(work, struct smd_port_t, connect_work.work);
	struct smd_port_info *port_info;
	int ret;

	port_info = port->port_info;

	pb_print(MSG_DEBUG, "port#%d\n", port->line);

	ret = smd_named_open_on_edge(port_info->name, SMD_APPS_MODEM,
				&port_info->ch, port, smd_notify);
	if (ret) {
		pb_print(MSG_DEBUG, "unable to open smd port:%s err:%d\n",
				port_info->name, ret);
		if (ret == -ENODEV) {
			pb_print(MSG_DEBUG, "Retry Open SMD port:%d\n",
					port->line);
			queue_delayed_work(smd_wq, &port->connect_work,
						msecs_to_jiffies(100));
		}
	} else {
		pb_print(MSG_INFO, "SMD port:%d Ready\n", port->line);
		smd_ready_flag = ((unsigned long)smd_ready_flag | (unsigned long)(0x01UL << port->line));
		if (((unsigned long)smd_ready_flag & 0x01UL) == 0x01UL) {
			set_function_up(SMD_SETTING);
		}
	}
}

static int smd_port_alloc(int line)
{
	struct smd_port_t *port;

	pb_print(MSG_DEBUG, "Entry\n");

	port = kzalloc(sizeof(struct smd_port_t), GFP_KERNEL);
	if (port == (struct smd_port_t *)NULL) {
		return -1;
	}

	port->ring = pb_ring_alloc_buf (PB_SMD_RING_BUF_SIZE);
	if (port->ring < 0) {
		kfree (port);
		return -1;
	}
	port->line = line;
	port->port_info = &smd_port_info[line];

	mutex_init(&port->lock);
	spin_lock_init(&port->port_lock);

	INIT_WORK(&port->tx, smd_tx);
	INIT_DELAYED_WORK(&port->connect_work, smd_connect_work);
	INIT_DELAYED_WORK(&port->smd_tx_retry, smd_tx_retry);

	smd_ports[line].port = port;

	queue_delayed_work(smd_wq, &port->connect_work, msecs_to_jiffies(10));

	pb_print(MSG_DEBUG, "port:%p line:%d\n", port, line);

	return 0;
}

static void smd_port_free(int line)
{
	pb_print(MSG_DEBUG, "Entry\n");

	if (!smd_ports[line].port){
		pb_print(MSG_ERROR, "pointer error\n");
	}else{
		pb_ring_free_buf (smd_ports[line].port->ring);
		kfree(smd_ports[line].port);
		smd_ports[line].port = NULL;
	}
}

void smd_send_data(char *buf, int len, int line)
{
	struct smd_port_t *port;
	unsigned int ret_busy;
	unsigned long	flags;

	if (!smd_ports[line].port) {
		pb_print(MSG_ERROR, "pointer error\n");
		return;
	}

	pb_print(MSG_DEBUG, "len=%d, line=%d, send data:%s\n", len, line, buf);
	port = smd_ports[line].port;

	spin_lock_irqsave(&port->port_lock, flags);
	if (!smd_put_ring(port, buf, len)){
		pb_print(MSG_DEBUG, "send work\n");
	}else{
		pb_print(MSG_ERROR, "ring buf over\n");
	}
	spin_unlock_irqrestore(&port->port_lock, flags);

	ret_busy = work_busy(&port->tx);
	if ( ret_busy == ((unsigned long)(WORK_BUSY_PENDING) | (unsigned long)(WORK_BUSY_RUNNING))){
		pb_print(MSG_DEBUG, "SMD send retry already pending and ruuning\n");
	}else if(ret_busy == (unsigned long)(WORK_BUSY_PENDING)){
		pb_print(MSG_DEBUG, "SMD send retry already pending\n");
	}else if(ret_busy == (unsigned long)(WORK_BUSY_RUNNING)){
		pb_print(MSG_DEBUG, "SMD send retry already running\n");
		queue_delayed_work(smd_wq, &port->smd_tx_retry,msecs_to_jiffies(5));
	}else{
		pb_print(MSG_DEBUG, "SMD send retry no pending and no running\n");
		queue_work(smd_wq, &port->tx);
	}
}

int smd_setup(void)
{
	int i, j;
	int ret = 0;

	pb_print(MSG_DEBUG, "Entry\n");

	smd_wq = create_workqueue("pb_smd_wq");

	if (!smd_wq) {
		pb_print(MSG_ERROR, "Unable to create workqueue smd_wq\n");
		return -1;
	}

	for (i = 0; i < SMD_PORT_NO; i++) {
		ret = smd_port_alloc(i);
		if (ret) {
			pb_print(MSG_ERROR, "alloc port error:%d\n", i);

#if defined(SMD_PORT_NO) && (1 < SMD_PORT_NO)
			for (j = 0; j < i; j++) {
				smd_port_free(j);
			}
#endif

			destroy_workqueue(smd_wq);
			smd_wq = NULL;
			ret = -1;
			break;
		}
	}

	return ret;
}

void smd_stop(void)
{
	int i;
	struct smd_port_t *port;
	pb_print(MSG_DEBUG, "Entry\n");

	if (smd_wq) {
		for (i = 0; i < SMD_PORT_NO; i++) {
			port = smd_ports[i].port;
			if (port) {
				cancel_delayed_work_sync(&port->connect_work);
				cancel_delayed_work_sync(&port->smd_tx_retry);
			}
		}
		flush_workqueue(smd_wq);
		destroy_workqueue(smd_wq);
		smd_wq = NULL;
	}
	for (i = 0; i < SMD_PORT_NO; i++) {
		smd_port_free(i);
	}
}

static void notify_usb_state(int state)
{
	static activity_type_e activity = PB_ACTIVITY_OFF;
	int is_send_to_modem = 0;
	pb_command_universal cmd;
	char cmd_buf [PB_COMMAND_ALL_LEN + 1];

	pb_print(MSG_DEBUG,"USB Notified me (%d)\n", state);

	switch (state)
	{
	case 0:
		if (activity == PB_ACTIVITY_ON)
		{
			is_send_to_modem = 1;
			activity = PB_ACTIVITY_OFF;
		}
		break;
	case 2:
		if (activity == PB_ACTIVITY_OFF)
		{
			is_send_to_modem = 1;
			activity = PB_ACTIVITY_ON;
		}
		break;
	case 1:
	case 3:
	case 4:
		break;
	default :
		pb_print(MSG_ERROR,"%s:Invalid USB Stattus(0x%04x)\n", __func__, state);
		break;
	}
	if (is_send_to_modem == 0L) {
		return;
	}

	cmd.id = PB_USB_ACTIVITY_ID;
	cmd.cmd.usba.activity = activity;
	if (PB_RESULT_OK != pb_encode_command (&cmd, cmd_buf))
	{
		pb_print(MSG_ERROR,"%s:Command encode failed\n", __func__);
		return;
	}

	smd_send_data(cmd_buf, PB_COMMAND_ALL_LEN, SMD_PORT_DATA4);
}
