/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2021 KYOCERA Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/errno.h>
#include <linux/device.h>
#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/regulator/consumer.h>
#include <linux/slab.h>
#include <sound/soc.h>
#include <sound/tlv.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/spinlock.h>
#include <linux/delay.h>

#include "tlv320aic3x.h"

/* /sys/module/tlv320aic3109/parameters/tlv320aic3109_reg_waddr  */
/* /sys/module/tlv320aic3109/parameters/tlv320aic3109_reg_raddr  */
/* /sys/module/tlv320aic3109/parameters/tlv320aic3109_reg_data   */
/* /sys/module/tlv320aic3109/parameters/tlv320aic3109_reset_gpio */
/* /sys/module/tlv320aic3109/parameters/tlv320aic3109_pwr_sync   */

static int set_reg(const char *val, struct kernel_param *kp);
static int get_reg(char *val, struct kernel_param *kp);

static unsigned int tlv320aic3109_pwr_sync = 1UL;
module_param(tlv320aic3109_pwr_sync, uint, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(tlv320aic3109_pwr_sync, "tlv320aic3109 power select sync or async");

static unsigned int tlv320aic3109_reg_waddr = 0UL;
module_param(tlv320aic3109_reg_waddr, uint, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(tlv320aic3109_reg_waddr, "tlv320aic3109 register address for Write");

static unsigned int tlv320aic3109_reg_raddr = 0UL;
module_param(tlv320aic3109_reg_raddr, uint, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(tlv320aic3109_reg_raddr, "tlv320aic3109 register address for Read");

static unsigned int tlv320aic3109_reg_data = 0UL;
module_param_call(tlv320aic3109_reg_data, set_reg, get_reg, &tlv320aic3109_reg_data, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(tlv320aic3109_reg_data, "tlv320aic3109 register R/W");

#define TLV320AIC3109_DEBUG
#ifdef TLV320AIC3109_DEBUG
static int set_gpio(const char *val, struct kernel_param *kp);
static int get_gpio(char *val, struct kernel_param *kp);
static unsigned int tlv320aic3109_reset_gpio = 0UL;
module_param_call(tlv320aic3109_reset_gpio, set_gpio, get_gpio, &tlv320aic3109_reset_gpio, S_IWUSR | S_IRUGO);
MODULE_PARM_DESC(tlv320aic3109_reset_gpio, "tlv320aic3109 gpio R/W");
#endif /* TLV320AIC3109_DEBUG */

static struct i2c_client *tlv320aic3109_client;

struct tlv320aic3109_data {
	struct mutex mutex;
	u8 power_state;
	struct kobject *tlv320aic3109_sysfs;
};

static void tlv320aic310_set_pinctrl(void)
{
#ifndef CONFIG_KYOCERA_MSND_NO_AUDIO
	struct pinctrl *pinctrl = NULL;
	struct pinctrl_state *gpio_state_active = NULL;
	int ret = 0L;

	if (WARN_ON(tlv320aic3109_client == NULL)) {
		return;
	}

	pinctrl = devm_pinctrl_get(&tlv320aic3109_client->dev);
	if (IS_ERR_OR_NULL(pinctrl)) {
		pr_err("%s: Pinctrl not defined", __func__);
	} else {
		gpio_state_active = pinctrl_lookup_state(pinctrl, PINCTRL_STATE_IDLE);
		if (IS_ERR_OR_NULL(gpio_state_active)) {
			pr_err("%s: Failed to lookup pinctrl for default state", __func__);
		} else {
			pr_info("%s: Pinctrl state is active.\n", __func__);
			ret = pinctrl_select_state(pinctrl, gpio_state_active);
			if (ret != 0L) {
				pr_err("%s: Failed to goto gpio-state=active: %d\n", __func__, ret);
			}
		}
	}
#endif /* CONFIG_KYOCERA_MSND_NO_AUDIO */
	return;
}

static int tlv320aic3109_power(u8 power)
{
#ifndef CONFIG_KYOCERA_MSND_NO_AUDIO
	struct tlv320aic3109_data *data = NULL;
	int codec_vcc_det = 0L;
	int codec_pwr_dwn = 0L;

	if (WARN_ON(tlv320aic3109_client == NULL)) {
		return -EINVAL;
	}

	data = i2c_get_clientdata(tlv320aic3109_client);

	codec_vcc_det = gpio_get_value(74U);
	codec_pwr_dwn = gpio_get_value(26U);
	pr_debug("%s: [B] CODEC_VCC_DET=%d, CODEC_PWR_DWN=%d\n", __func__, codec_vcc_det, codec_pwr_dwn);

	if (tlv320aic3109_pwr_sync == 1UL) {
		if (codec_vcc_det == codec_pwr_dwn) {
			return 0L;
		}
		if (codec_vcc_det == 1L) {
			mdelay(5UL);
			(void)gpio_direction_output(26U, 1L); /* L -> H */
			(void)i2c_smbus_read_byte_data(tlv320aic3109_client, AIC3X_PAGE_SELECT);
			data->power_state = 1U;
		} else {
			(void)gpio_direction_output(26U, 0L); /* H -> L */
			data->power_state = 0U;
		}
	} else {
		data->power_state = power;
		if ((power == 1U) && (codec_pwr_dwn == 0L)) {
			mdelay(5UL);
			(void)gpio_direction_output(26U, 1L); /* L -> H */
			(void)i2c_smbus_read_byte_data(tlv320aic3109_client, AIC3X_PAGE_SELECT);
		}
		if ((power == 0U) && (codec_pwr_dwn == 1L)) {
			(void)gpio_direction_output(26U, 0L); /* H -> L */
		}
	}

	codec_vcc_det = gpio_get_value(74U);
	codec_pwr_dwn = gpio_get_value(26U);
	pr_debug("%s: [A] CODEC_VCC_DET=%d, CODEC_PWR_DWN=%d\n", __func__, codec_vcc_det, codec_pwr_dwn);

	return 0L;
#else
	return -ENODEV;
#endif /* CONFIG_KYOCERA_MSND_NO_AUDIO */
}

static int tlv320aic3109_i2c_read(int reg)
{
#ifndef CONFIG_KYOCERA_MSND_NO_AUDIO
	struct tlv320aic3109_data *data = NULL;
	int val = 0L;

	if (WARN_ON(tlv320aic3109_client == NULL)) {
		return -EINVAL;
	}

	data = i2c_get_clientdata(tlv320aic3109_client);

	mutex_lock(&data->mutex);

	val = tlv320aic3109_power(1U);
	if (val == 0L) {
		if (data->power_state == 1U) {
			val = i2c_smbus_read_byte_data(tlv320aic3109_client, reg);
			if (val < 0L) {
				pr_err("%s: Failed to read. (addr:0x%02x)\n", __func__, (unsigned int)reg);
			}
		} else {
			pr_debug("%s: power_state is off.\n", __func__);
		}
	}

	mutex_unlock(&data->mutex);

	return val;
#else
	return -ENODEV;
#endif /* CONFIG_KYOCERA_MSND_NO_AUDIO */
}

static int tlv320aic3109_i2c_write(int reg, u8 value)
{
#ifndef CONFIG_KYOCERA_MSND_NO_AUDIO
	struct tlv320aic3109_data *data = NULL;
	int val = 0L;

	if (WARN_ON(tlv320aic3109_client == NULL)) {
		return -EINVAL;
	}

	data = i2c_get_clientdata(tlv320aic3109_client);

	mutex_lock(&data->mutex);

	val = tlv320aic3109_power(1U);
	if (val == 0L) {
		if (data->power_state == 1U) {
			val = i2c_smbus_write_byte_data(tlv320aic3109_client, reg, value);
			if (val < 0L) {
				pr_err("%s: Failed to write. (addr:0x%02x)\n", __func__, (unsigned int)reg);
			}
		} else {
			pr_debug("%s: power_state is off.\n", __func__);
		}
	}

	mutex_unlock(&data->mutex);

	return val;
#else
	return -ENODEV;
#endif /* CONFIG_KYOCERA_MSND_NO_AUDIO */
}

static int get_reg(char *val, struct kernel_param *kp)
{
	int rc = 0L;

	rc = tlv320aic3109_i2c_read(tlv320aic3109_reg_raddr);
	if (rc < 0L) {
		return rc;
	}

	return sprintf(val, "0x%02X", (unsigned int)rc);
}

static int set_reg(const char *val, struct kernel_param *kp)
{
	int rc = 0L;

	rc = param_set_ulong(val, kp);
	if (rc != 0L) {
		return rc;
	}

	rc = tlv320aic3109_i2c_write(tlv320aic3109_reg_waddr, (u8)tlv320aic3109_reg_data);
	if (rc < 0L) {
		return rc;
	}

	return 0L;
}

#ifdef TLV320AIC3109_DEBUG
static int get_gpio(char *val, struct kernel_param *kp)
{
	int rc = 0L;

	rc = gpio_get_value(26U);

	return sprintf(val, "0x%02X", (unsigned int)rc);
}

static int set_gpio(const char *val, struct kernel_param *kp)
{
	int rc = 0L;

	rc = param_set_ulong(val, kp);
	if (rc != 0L) {
		return rc;
	}

	(void)gpio_direction_output(26U, tlv320aic3109_reset_gpio);

	return 0L;
}
#endif /* TLV320AIC3109_DEBUG */

#define CODEC_WRITE_MAX_PRM 3UL
#define CODEC_POWER_ON      1UL
#define CODEC_POWER_OFF     0UL

static ssize_t tlv320aic3109_sysfs_codec_write(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	char  copy[128] = { 0 };
	char *tmp = NULL;
	char *token = copy;
	int   r_data = 0L;
	int   ret = 0L;
	uint  cnt = 0UL;
	uint  prm[3] = { 0UL };
	uint  w_data = 0UL;

	strncpy(copy, buf, sizeof(copy));
	do {
		tmp = strsep(&token, ",");
		if (tmp == NULL) {
			ret = -EINVAL;
			break;
		}
		ret = kstrtouint(tmp, 0UL, &prm[cnt++]);
		if (ret != 0L) {
			break;
		}
	} while (cnt < CODEC_WRITE_MAX_PRM);

	pr_debug("%s: addr:0x%x, value:0x%x, mask:0x%x, ret=%d\n", __func__, prm[0], prm[1], prm[2], ret);

	if (ret < 0L) {
		pr_err("%s: invalid arguments. ret=%d\n", __func__, ret);
		return ret;
	}

	r_data = tlv320aic3109_i2c_read(prm[0]);
	if (r_data < 0L) {
		return r_data;
	}

	w_data  =  (prm[1] &  prm[2]);
	r_data  =  (r_data & ~prm[2]);
	w_data  =  (w_data |  r_data);

	pr_debug("%s: w_data: 0x%x, r_data: 0x%x\n", __func__, w_data, r_data);

	ret = tlv320aic3109_i2c_write(prm[0], (u8)w_data);
	if (ret < 0L) {
		return ret;
	}

	return count;
}

static DEVICE_ATTR(codec_write, S_IWUSR | S_IWGRP, NULL, tlv320aic3109_sysfs_codec_write);

static ssize_t tlv320aic3109_show_power_status(struct device *dev, struct device_attribute *attr, char *buf)
{
	int pwr_chk_1 = 0L;
	int pwr_chk_2 = 0L;

	pwr_chk_1 = tlv320aic3109_i2c_read(LINE1L_2_LADC_CTRL);
	pwr_chk_2 = tlv320aic3109_i2c_read(DAC_PWR);

	if ((pwr_chk_1 == 0x7CL) && (pwr_chk_2 == LDAC_PWR_ON)) {
		pr_debug("%s: Audio codec ic power is on.\n", __func__);
		return sprintf(buf, "%lu", CODEC_POWER_ON);
	}

	pr_debug("%s: Audio codec ic power is off.\n", __func__);
	return sprintf(buf, "%lu", CODEC_POWER_OFF);
}

static DEVICE_ATTR(power_status, S_IRUGO, tlv320aic3109_show_power_status, NULL);

static struct attribute *tlv320aic3109_sysfs_codec_attributes[] = {
	&dev_attr_codec_write.attr,
	&dev_attr_power_status.attr,
	NULL
};

static const struct attribute_group tlv320aic3109_sysfs_codec_attr_group = {
	.attrs = tlv320aic3109_sysfs_codec_attributes,
};

static int tlv320aic3109_sysfs_init(void)
{
	struct tlv320aic3109_data *data = NULL;
	int ret = 0L;

	if (WARN_ON(tlv320aic3109_client == NULL)) {
		return -EINVAL;
	}

	data = i2c_get_clientdata(tlv320aic3109_client);

	data->tlv320aic3109_sysfs = kobject_create_and_add("tlv320aic3109_sysfs", kernel_kobj);
	if (data->tlv320aic3109_sysfs == NULL) {
		pr_err("%s: kobject_create_and_add error\n", __func__);
		return -ENOMEM;
	}

	ret = sysfs_create_group(data->tlv320aic3109_sysfs, &tlv320aic3109_sysfs_codec_attr_group);
	if (ret < 0L) {
		pr_err("%s: sysfs_create_group error %d\n", __func__, ret);
		kobject_put(data->tlv320aic3109_sysfs);
	}
	
	return ret;
}

static int tlv320aic3109_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
	struct tlv320aic3109_data *data = NULL;

	data = devm_kzalloc(&client->dev, sizeof(*data), GFP_KERNEL);
	if (data == NULL) {
		return -ENOMEM;
	}

	tlv320aic3109_client = client;

	i2c_set_clientdata(tlv320aic3109_client, data);

	mutex_init(&data->mutex);

	tlv320aic3109_client->adapter->retries = 3L;
	tlv320aic310_set_pinctrl();
	tlv320aic3109_sysfs_init();

	return 0L;
}

static int tlv320aic3109_remove(struct i2c_client *client)
{
	struct tlv320aic3109_data *data = NULL;

	if (WARN_ON(!tlv320aic3109_client))
		return -EINVAL;

	data = i2c_get_clientdata(tlv320aic3109_client);

	sysfs_remove_group(data->tlv320aic3109_sysfs, &tlv320aic3109_sysfs_codec_attr_group);
	kobject_put(data->tlv320aic3109_sysfs);

	tlv320aic3109_power(0U);
	tlv320aic3109_client = NULL;

	return 0L;
}

static const struct i2c_device_id tlv320aic3109_id[2] = {
	{ "tlv320aic3109", 0UL },
	{}
};
MODULE_DEVICE_TABLE(i2c, tlv320aic3109_id);

static const struct of_device_id tlv320aic3109_of_match[2] = {
	{ .compatible = "ti,tlv320aic3109", },
	{},
};
MODULE_DEVICE_TABLE(of, tlv320aic3109_of_match);

static struct i2c_driver tlv320aic3109_i2c_driver = {
	.driver = {
		.name           = "tlv320aic3109-codec",
		.owner          = THIS_MODULE,
		.of_match_table = of_match_ptr(tlv320aic3109_of_match),
	},
	.probe    = tlv320aic3109_probe,
	.remove   = tlv320aic3109_remove,
	.id_table = tlv320aic3109_id,
};

module_i2c_driver(tlv320aic3109_i2c_driver);

MODULE_AUTHOR("KYOCERA Corporation");
MODULE_DESCRIPTION("Audio Codec Driver for TLV320AIC3109");
MODULE_LICENSE("GPL");
