/*
 * Copyright (c) 2011-2014, The Linux Foundation. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2014 KYOCERA Corporation
 * (C) 2016 KYOCERA Corporation
 * (C) 2019 KYOCERA Corporation
 */

#include <linux/kernel.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/slab.h>
#include <linux/termios.h>
#include <soc/qcom/smd.h>
#include <linux/debugfs.h>
#include <linux/bitops.h>
#include <linux/termios.h>

#include <linux/module.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>

#include "u_rmnet.h"
#include "kcusb_log.h"

#define MAX_CTRL_PER_CLIENT_TTY	1
#define MAX_CTRL_PORT_TTY		(MAX_CTRL_PER_CLIENT_TTY * NR_CTRL_CLIENTS)
static struct workqueue_struct *grmnet_ctrl_tty_wq;
static struct tty_driver *grmnet_ctrl_driver;

u8 online_clients_tty;

#define OFFLINE_UL_Q_LIMIT_TTY	1000

static unsigned int offline_ul_ctrl_pkt_q_limit_tty = OFFLINE_UL_Q_LIMIT_TTY;
module_param(offline_ul_ctrl_pkt_q_limit_tty, uint, S_IRUGO | S_IWUSR);

#define TTY_CH_OPENED	0
#define WRITE_BUF_SIZE   8192

struct grmnet_ctrl_buf {
	unsigned buf_size;
	char     *buf_buf;
	char     *buf_get;
	char     *buf_put;
};

struct tty_ch_info {
	char			*name;
	unsigned long		flags;
	wait_queue_head_t tty_wait_q;

	struct list_head	tx_q;
	unsigned long		tx_len;

	struct work_struct	write_w;

	struct rmnet_ctrl_port_tty	*port;
};

struct rmnet_ctrl_port_tty {
	struct tty_ch_info	ctrl_ch;
	unsigned int		port_num;
	struct grmnet		*port_usb;
	spinlock_t		port_lock;

	bool   openclose;
	struct tty_port port;
	struct usb_cdc_line_coding port_line_coding;
	wait_queue_head_t	drain_wait;	/* wait while writes drain */
};

static struct rmnet_ctrl_ports_tty {
	struct mutex lock;
	struct rmnet_ctrl_port_tty *port;
} ctrl_tty_ports[MAX_CTRL_PORT_TTY];

static void grmnet_ctrl_tty_notify(void *p, unsigned event);

static int grmnet_ctrl_open(struct tty_struct *tty, struct file *file)
{
	int port_num = tty->index;
	int status   = 0;
	struct rmnet_ctrl_port_tty *port = NULL;
	
	do {
		mutex_lock(&ctrl_tty_ports[port_num].lock);
		port = ctrl_tty_ports[port_num].port;
		if (!port)
			status = -ENODEV;
		else {
			spin_lock_irq(&port->port_lock);

			/* already open?  Great. */
			if (port->port.count) {
				status = 0;
				port->port.count++;

			/* currently opening/closing? wait ... */
			} else if (port->openclose) {
				status = -EBUSY;

			/* ... else we do the work */
			} else {
				status = -EAGAIN;
				port->openclose = true;
			}
			spin_unlock_irq(&port->port_lock);
		}
		mutex_unlock(&ctrl_tty_ports[port_num].lock);

		switch (status) {
		default:
			/* fully handled */
			return status;
		case -EAGAIN:
			/* must do the work */
			break;
		case -EBUSY:
			/* wait for EAGAIN task to finish */
			msleep(1);
			/* REVISIT could have a waitchannel here, if
			 * concurrent open performance is important
			 */
			break;
		}
	} while (status != -EAGAIN);

	/* Do the "real open" */
	spin_lock_irq(&port->port_lock);

	set_bit(TTY_NO_WRITE_SPLIT,&tty->flags);

	/* REVISIT if REMOVED (ports[].port NULL), abort the open
	 * to let rmmod work faster (but this way isn't wrong).
	 */

	/* REVISIT maybe wait for "carrier detect" */

	tty->driver_data = port;
	port->port.tty = tty;

	port->port.count = 1;
	port->openclose = false;

	/* if connected, start the I/O stream */
	if (port->port_usb) {
		grmnet_ctrl_tty_notify(port, SMD_EVENT_OPEN);
	}

	pr_debug("%s: ttyDM%d (%p,%p)\n", __func__, port->port_num, tty, file);

	status = 0;

	spin_unlock_irq(&port->port_lock);
	return status;
}

static void grmnet_ctrl_close(struct tty_struct *tty, struct file *file)
{
	struct rmnet_ctrl_port_tty	*port = tty->driver_data;

	spin_lock_irq(&port->port_lock);

	if (port->port.count != 1) {
		if (port->port.count == 0)
			WARN_ON(1);
		else
			--port->port.count;
		spin_unlock_irq(&port->port_lock);
		return;
	}

	pr_debug("%s: ttyDM%d (%p,%p) ...\n", __func__, port->port_num, tty, file);

	/* mark port as closing but in use; we can drop port lock
	 * and sleep if necessary
	 */
	port->openclose = true;
	port->port.count = 0;
	
	clear_bit(TTY_NO_WRITE_SPLIT,&tty->flags);
	
	tty->driver_data = NULL;
	port->port.tty = NULL;

	port->openclose = false;

	pr_debug("%s: ttyDM%d (%p,%p) done!\n",
			__func__, port->port_num, tty, file);

	wake_up(&port->port.close_wait);
	spin_unlock_irq(&port->port_lock);

	grmnet_ctrl_tty_notify(port, SMD_EVENT_CLOSE);
}

static int grmnet_ctrl_write(struct tty_struct *tty, const unsigned char *buf, int count)
{
	struct rmnet_ctrl_port_tty *port = tty->driver_data;
	unsigned long	flags;

	if (!port)
		return 0;
	pr_debug("%s: ttyDM%d (%p) writing %d bytes\n",
			__func__, port->port_num, tty, count);

	USBLOGOUT("DM: sd\n");
	spin_lock_irqsave(&port->port_lock, flags);
	if (count && port->port_usb && port->port_usb->send_cpkt_response) {
		USBLOGOUT("DM: tr%d\n", count);
		port->port_usb->send_cpkt_response(port->port_usb, (void*)buf, count);
	}
	spin_unlock_irqrestore(&port->port_lock, flags);

	return count;
}

static int grmnet_ctrl_write_room(struct tty_struct *tty)
{
	struct rmnet_ctrl_port_tty *port = tty->driver_data;
	unsigned long flags;
	int room = 0;
	
	if (!port)
		return 0;
	
	spin_lock_irqsave(&port->port_lock, flags);
	if (port->port_usb)
		room = WRITE_BUF_SIZE;
	spin_unlock_irqrestore(&port->port_lock, flags);

	pr_debug("%s: (%d,%p) room=%d\n",
		__func__, port->port_num, tty, room);

	return room;
}

static const struct tty_operations gs_tty_ops = {
	.open       = grmnet_ctrl_open,
	.close      = grmnet_ctrl_close,
	.write      = grmnet_ctrl_write,
	.write_room = grmnet_ctrl_write_room,
};

static int gtty_ctrl_tty_init(void)
{
	unsigned i;
	int status;
	
	grmnet_ctrl_driver = alloc_tty_driver(MAX_CTRL_PORT_TTY);
	if (!grmnet_ctrl_driver)
		return -ENOMEM;
	
	grmnet_ctrl_driver->driver_name = "tty_ctrl_tty";
	grmnet_ctrl_driver->name        = "ttyDM";
	
	grmnet_ctrl_driver->type         = TTY_DRIVER_TYPE_SERIAL;
	grmnet_ctrl_driver->subtype      = SERIAL_TYPE_NORMAL;
	grmnet_ctrl_driver->flags        = TTY_DRIVER_REAL_RAW | TTY_DRIVER_DYNAMIC_DEV | TTY_DRIVER_RESET_TERMIOS;
	grmnet_ctrl_driver->init_termios = tty_std_termios;
	
	grmnet_ctrl_driver->init_termios.c_cflag = B9600 | CS8 | CREAD | HUPCL | CLOCAL;
	grmnet_ctrl_driver->init_termios.c_iflag = 0;
	grmnet_ctrl_driver->init_termios.c_lflag = 0;
	grmnet_ctrl_driver->init_termios.c_oflag = 0;
	
	grmnet_ctrl_driver->init_termios.c_ispeed = 9600;
	grmnet_ctrl_driver->init_termios.c_ospeed = 9600;
	
	tty_set_operations(grmnet_ctrl_driver, &gs_tty_ops);
	for (i = 0; i < MAX_CTRL_PORT_TTY; i++)
		mutex_init(&ctrl_tty_ports[i].lock);
	
	status = tty_register_driver(grmnet_ctrl_driver);
	if (status) {
		pr_err("%s: cannot register, err %d\n", __func__, status);
		goto fail;
	}
	
	pr_debug("%s: registered %d ttyDM* device%s\n", __func__, MAX_CTRL_PORT_TTY, (MAX_CTRL_PORT_TTY == 1) ? "" : "s");
	return status;
fail:
	put_tty_driver(grmnet_ctrl_driver);
	grmnet_ctrl_driver = NULL;
	return status;
}

static int gtty_ctrl_tty_exit(void)
{
	tty_unregister_driver(grmnet_ctrl_driver);
	put_tty_driver(grmnet_ctrl_driver);
	grmnet_ctrl_driver = NULL;
	return 0;
}

void gtty_ctrl_read_pending(u8 portno)
{
	return;
}

/*---------------misc functions---------------- */

static struct rmnet_ctrl_pkt *alloc_rmnet_ctrl_pkt_tty(unsigned len, gfp_t flags)
{
	struct rmnet_ctrl_pkt *pkt;

	pkt = kzalloc(sizeof(struct rmnet_ctrl_pkt), flags);
	if (!pkt)
		return ERR_PTR(-ENOMEM);

	pkt->buf = kmalloc(len, flags);
	if (!pkt->buf) {
		kfree(pkt);
		return ERR_PTR(-ENOMEM);
	}

	pkt->len = len;

	return pkt;
}

static void free_rmnet_ctrl_pkt_tty(struct rmnet_ctrl_pkt *pkt)
{
	kfree(pkt->buf);
	kfree(pkt);
}

/*--------------------------------------------- */

/*---------------control/tty channel functions---------------- */

static void grmnet_ctrl_tty_write_w(struct work_struct *w)
{
	struct tty_ch_info *c = container_of(w, struct tty_ch_info, write_w);
	struct rmnet_ctrl_port_tty *port = c->port;
	unsigned long flags;
	struct rmnet_ctrl_pkt *cpkt;
	int ret;

	spin_lock_irqsave(&port->port_lock, flags);
	while (port->port.count) {
		if (list_empty(&c->tx_q))
			break;

		cpkt = list_first_entry(&c->tx_q, struct rmnet_ctrl_pkt, list);

		if (tty_buffer_space_avail(&port->port) < cpkt->len) {
			USBLOGOUT("DM: rbc%d\n", cpkt->len);
		} else {
			spin_unlock_irqrestore(&port->port_lock, flags);
			ret = tty_insert_flip_string(&port->port, cpkt->buf, cpkt->len);
			spin_lock_irqsave(&port->port_lock, flags);
			USBLOGOUT("DM: tw%d\n", ret);
			if (ret != cpkt->len) {
				pr_err("%s: TTY_write failed err:%d\n", __func__, ret);
				USBLOGOUT("DM: rbc%d\n", cpkt->len);
			}
		}

		list_del(&cpkt->list);
		free_rmnet_ctrl_pkt_tty(cpkt);
		tty_flip_buffer_push(&port->port);
	}
	spin_unlock_irqrestore(&port->port_lock, flags);
}

static int is_legal_port_num_tty(u8 portno)
{
	if (portno >= MAX_CTRL_PORT_TTY)
		return false;
	if (ctrl_tty_ports[portno].port == NULL)
		return false;

	return true;
}

static int
grmnet_ctrl_tty_send_cpkt_tomodem(u8 portno,
	void *buf, size_t len)
{
	unsigned long		flags;
	struct rmnet_ctrl_port_tty	*port;
	struct tty_ch_info	*c;
	struct rmnet_ctrl_pkt *cpkt;

	if (!is_legal_port_num_tty(portno)) {
		pr_err("%s: Invalid portno#%d\n", __func__, portno);
		return -ENODEV;
	}

	port = ctrl_tty_ports[portno].port;

	cpkt = alloc_rmnet_ctrl_pkt_tty(len, GFP_ATOMIC);
	if (IS_ERR(cpkt)) {
		pr_err("%s: Unable to allocate ctrl pkt\n", __func__);
		return -ENOMEM;
	}

	memcpy(cpkt->buf, buf, len);
	cpkt->len = len;

	spin_lock_irqsave(&port->port_lock, flags);
	c = &port->ctrl_ch;

	/* queue cpkt if ch is not open, would be sent once ch is opened */
	if (!test_bit(TTY_CH_OPENED, &c->flags)) {
		USBLOGOUT("DM: rbc%d\n", cpkt->len);
		free_rmnet_ctrl_pkt_tty(cpkt);
		spin_unlock_irqrestore(&port->port_lock, flags);
		return 0;
	}

	list_add_tail(&cpkt->list, &c->tx_q);
	queue_work(grmnet_ctrl_tty_wq, &c->write_w);
	spin_unlock_irqrestore(&port->port_lock, flags);

	return 0;
}

static char *get_smd_event_tty(unsigned event)
{
	switch (event) {
		case SMD_EVENT_OPEN:
		return "OPEN";
	case SMD_EVENT_CLOSE:
		return "CLOSE";
	}

	return "UNDEFINED";
}

static void grmnet_ctrl_tty_notify(void *p, unsigned event)
{
	struct rmnet_ctrl_port_tty	*port = p;
	struct tty_ch_info	*c = &port->ctrl_ch;
	struct rmnet_ctrl_pkt	*cpkt;
	unsigned long		flags;
	int avail;

	pr_debug("%s: EVENT_(%s)\n", __func__, get_smd_event_tty(event));

	switch (event) {
		case SMD_EVENT_OPEN:
		set_bit(TTY_CH_OPENED, &c->flags);
		USBLOGOUT("DM: so\n");

		if (port && port->port_usb && port->port_usb->connect)
			port->port_usb->connect(port->port_usb);

		if (port) {
			/* Send data to modem incase already received over USB */
			avail = tty_buffer_space_avail(&port->port);
			if (avail) {
				pr_debug("%s: swa%d\n", __func__, avail);
				queue_work(grmnet_ctrl_tty_wq, &c->write_w);
			}
		}
		break;
	case SMD_EVENT_CLOSE:
		clear_bit(TTY_CH_OPENED, &c->flags);
		USBLOGOUT("DM: sc\n");

		if (port && port->port_usb && port->port_usb->disconnect)
			port->port_usb->disconnect(port->port_usb);

		spin_lock_irqsave(&port->port_lock, flags);
		while (!list_empty(&c->tx_q)) {
			cpkt = list_first_entry(&c->tx_q,
					struct rmnet_ctrl_pkt, list);

			list_del(&cpkt->list);
			USBLOGOUT("DM: rbc%d\n", cpkt->len);
			free_rmnet_ctrl_pkt_tty(cpkt);
		}
		spin_unlock_irqrestore(&port->port_lock, flags);

		break;
	}
	wake_up(&c->tty_wait_q);
}
/*------------------------------------------------------------ */

int gtty_ctrl_connect(struct grmnet *gr, int port_num)
{
	struct rmnet_ctrl_port_tty	*port;
	unsigned long		flags;

	pr_debug("%s: grmnet:%p port#%d\n", __func__, gr, port_num);

	if (!is_legal_port_num_tty(port_num)) {
		pr_err("%s: Invalid port_num#%d\n", __func__, port_num);
		return -ENODEV;
	}

	if (!gr) {
		pr_err("%s: grmnet port is null\n", __func__);
		return -ENODEV;
	}

	port = ctrl_tty_ports[port_num].port;

	spin_lock_irqsave(&port->port_lock, flags);
	port->port_usb = gr;
	gr->send_encap_cmd = grmnet_ctrl_tty_send_cpkt_tomodem;
	spin_unlock_irqrestore(&port->port_lock, flags);

	if (0 < port->port.count) {
		grmnet_ctrl_tty_notify(port, SMD_EVENT_OPEN);
	}
	
	return 0;
}

void gtty_ctrl_disconnect(struct grmnet *gr, u8 port_num)
{
	struct rmnet_ctrl_port_tty	*port;
	unsigned long		flags;
	struct tty_ch_info	*c;
	struct rmnet_ctrl_pkt	*cpkt;

	pr_debug("%s: grmnet:%p port#%d\n", __func__, gr, port_num);

	if (!is_legal_port_num_tty(port_num)) {
		pr_err("%s: Invalid port_num#%d\n", __func__, port_num);
		return;
	}

	if (!gr) {
		pr_err("%s: grmnet port is null\n", __func__);
		return;
	}

	port = ctrl_tty_ports[port_num].port;
	c = &port->ctrl_ch;
	cancel_work_sync(&c->write_w);

	spin_lock_irqsave(&port->port_lock, flags);
	port->port_usb = 0;
	gr->send_encap_cmd = 0;

	while (!list_empty(&c->tx_q)) {
		cpkt = list_first_entry(&c->tx_q, struct rmnet_ctrl_pkt, list);

		list_del(&cpkt->list);
		USBLOGOUT("DM: rbc%d\n", cpkt->len);
		free_rmnet_ctrl_pkt_tty(cpkt);
	}

	spin_unlock_irqrestore(&port->port_lock, flags);

	clear_bit(TTY_CH_OPENED, &c->flags);
}

static void grmnet_ctrl_tty_port_free(int portno)
{
	struct rmnet_ctrl_port_tty	*port = ctrl_tty_ports[portno].port;

	if (port) {
		kfree(port);
	}
}

static int grmnet_ctrl_tty_port_alloc(int portno, struct usb_cdc_line_coding *coding)
{
	struct rmnet_ctrl_port_tty	*port;
	struct tty_ch_info	*c;
	int		ret = 0;

	if (portno >= MAX_CTRL_PORT_TTY) {
		pr_err("Illegal port number.\n");
		return -EINVAL;
	}

	mutex_lock(&ctrl_tty_ports[portno].lock);
	if (ctrl_tty_ports[portno].port) {
		pr_err("serial line %d is in use.\n", portno);
		ret = -EBUSY;
		goto out;
	}

	port = kzalloc(sizeof(struct rmnet_ctrl_port_tty), GFP_KERNEL);
	if (!port) {
		ret = -ENOMEM;
		goto out;
	}

	tty_port_init(&port->port);
	tty_buffer_set_limit(&port->port, 131072);
	spin_lock_init(&port->port_lock);
	init_waitqueue_head(&port->drain_wait);

	c = &port->ctrl_ch;
	c->port = port;

	init_waitqueue_head(&c->tty_wait_q);
	INIT_LIST_HEAD(&c->tx_q);
	INIT_WORK(&c->write_w, grmnet_ctrl_tty_write_w);

	port->port_num = portno;
	port->port_line_coding = *coding;

	ctrl_tty_ports[portno].port = port;

	pr_debug("%s: port:%p portno:%d\n", __func__, port, portno);

out:
	mutex_unlock(&ctrl_tty_ports[portno].lock);
	return ret;
}

int gtty_ctrl_setup(enum ctrl_client client_num, unsigned int count,
					u8 *first_port_idx)
{
	int	i, start_port, allocated_ports;
	int	ret;
	struct usb_cdc_line_coding coding;
	struct device *tty_dev;

	coding.dwDTERate = cpu_to_le32(9600);
	coding.bCharFormat = 8;
	coding.bParityType = USB_CDC_NO_PARITY;
	coding.bDataBits = USB_CDC_1_STOP_BITS;

	pr_debug("%s: requested ports:%d\n", __func__, count);

	if (client_num >= NR_CTRL_CLIENTS) {
		pr_err("%s: Invalid client:%d\n", __func__, client_num);
		return -EINVAL;
	}

	if (!count || count > MAX_CTRL_PER_CLIENT_TTY) {
		pr_err("%s: Invalid num of ports count:%d\n",
				__func__, count);
		return -EINVAL;
	}

	if (!online_clients_tty) {
		grmnet_ctrl_tty_wq = alloc_workqueue("gtty_ctrl",
			WQ_UNBOUND | WQ_MEM_RECLAIM, 1);
		if (!grmnet_ctrl_tty_wq) {
			pr_err("%s: Unable to create workqueue grmnet_ctrl\n",
					__func__);
			return -ENOMEM;
		}
	}
	online_clients_tty++;

	start_port = MAX_CTRL_PER_CLIENT_TTY * client_num;
	allocated_ports = 0;
	for (i = start_port; i < count + start_port; i++) {
		allocated_ports++;
		ret = grmnet_ctrl_tty_port_alloc(i, &coding);
		if (ret) {
			pr_err("%s: Unable to alloc port:%d\n", __func__, i);
			allocated_ports--;
			goto free_ctrl_tty_ports;
		}
		tty_dev = tty_port_register_device(&ctrl_tty_ports[i].port->port,
				grmnet_ctrl_driver, i, NULL);
		
		if (IS_ERR(tty_dev)) {
			pr_err("%s: failed to register tty for port %d, err %ld\n",
					__func__, i, PTR_ERR(tty_dev));
			allocated_ports--;
			tty_port_destroy(&ctrl_tty_ports[i].port->port);
			goto free_ctrl_tty_ports;
		}
	}
	if (first_port_idx)
		*first_port_idx = start_port;
	return 0;

free_ctrl_tty_ports:
	for (i = 0; i < allocated_ports; i++)
		grmnet_ctrl_tty_port_free(start_port + i);


	online_clients_tty--;
	if (!online_clients_tty)
		destroy_workqueue(grmnet_ctrl_tty_wq);

	return ret;
}

static int __init gtty_ctrl_init(void)
{
	gtty_ctrl_tty_init();
	online_clients_tty = 0;

	return 0;
}
module_init(gtty_ctrl_init);

static void __exit gtty_ctrl_exit(void)
{
	gtty_ctrl_tty_exit();
}

static int gs_closed_tty(struct rmnet_ctrl_port_tty *port)
{
	int cond;

	spin_lock_irq(&port->port_lock);
	cond = (port->port.count == 0) && !port->openclose;
	spin_unlock_irq(&port->port_lock);
	return cond;
}

static void gserial_free_port_tty(struct rmnet_ctrl_port_tty *port)
{
	/* wait for old opens to finish */
	wait_event(port->port.close_wait, gs_closed_tty(port));
	WARN_ON(port->port_usb != NULL);
	tty_port_destroy(&port->port);
	kfree(port);
}

void gserial_free_line_tty(unsigned char port_num)
{
	struct rmnet_ctrl_port_tty	*port;

	mutex_lock(&ctrl_tty_ports[port_num].lock);
	if (WARN_ON(!ctrl_tty_ports[port_num].port)) {
		mutex_unlock(&ctrl_tty_ports[port_num].lock);
		return;
	}
	port = ctrl_tty_ports[port_num].port;
	ctrl_tty_ports[port_num].port = NULL;
	mutex_unlock(&ctrl_tty_ports[port_num].lock);

	gserial_free_port_tty(port);
	tty_unregister_device(grmnet_ctrl_driver, port_num);
}
EXPORT_SYMBOL_GPL(gserial_free_line_tty);
module_exit(gtty_ctrl_exit);
MODULE_DESCRIPTION("tty control driver");
MODULE_LICENSE("GPL v2");
