/*
 * pb_ring.c
 *
 * this program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful.
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 */
/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2014 KYOCERA Corporation
 * (C) 2018 KYOCERA Corporation
 * (C) 2019 KYOCERA Corporation
 */

#include <linux/slab.h>

#include "pb_serial.h"
#include "pb_ring.h"

#define PB_RING_DEBUG_PREC_CHECK
#define PB_MAX_NUM_OF_INSTANCE	8
struct pb_ring {
	int head;
	int tail;
	int size;
	char* buf;
};

static struct pb_ring rings [PB_MAX_NUM_OF_INSTANCE];
int pb_ring_alloc_buf (const int size) {
	int i;
	struct pb_ring* ring;
	for (i = 0; i < PB_MAX_NUM_OF_INSTANCE; i++) {
		ring = &rings[i];
		if (ring->size == 0) {
			ring->buf = kzalloc(size, GFP_KERNEL);
			if (!ring->buf) {
				pb_print(MSG_ERROR, "kzalloc error\n");
				return -1;
			}
			ring->size = size;
			ring->head = ring->tail = 0;
			return i;
		}
	}
	pb_print(MSG_ERROR, "Ring max exceeded\n");
	return -1;
}

void pb_ring_free_buf (const int handle) {
	#if defined  PB_RING_DEBUG_PREC_CHECK
	if (handle < 0 || PB_MAX_NUM_OF_INSTANCE <= handle) {
		pb_print(MSG_ERROR, "Invalid handle is specified(%d)\n", handle);
		return;
	}
	#endif /* PB_RING_DEBUG_PREC_CHECK */
	kfree (rings [handle].buf);
	rings [handle].size = 0;
	rings [handle].buf  = 0;
}

int pb_ring_is_empty (const int handle) {
	#if defined  PB_RING_DEBUG_PREC_CHECK
	if (handle < 0 || PB_MAX_NUM_OF_INSTANCE <= handle) {
		pb_print(MSG_ERROR, "Invalid handle is specified(%d)\n", handle);
		return -1;
	}
	#endif /* PB_RING_DEBUG_PREC_CHECK */
	return rings [handle].head == rings [handle].tail;
}

extern int pb_ring_empty_buf (const int handle) {
	#if defined  PB_RING_DEBUG_PREC_CHECK
	if (handle < 0 || PB_MAX_NUM_OF_INSTANCE <= handle) {
		pb_print(MSG_ERROR, "Invalid handle is specified(%d)\n", handle);
		return -1;
	}
	#endif /* PB_RING_DEBUG_PREC_CHECK */
	rings [handle].head = rings [handle].tail = 0;
	return handle;
}

int pb_ring_is_enough (const int handle, const int size) {
	int used_size;
	#if defined  PB_RING_DEBUG_PREC_CHECK
	if (handle < 0 || PB_MAX_NUM_OF_INSTANCE <= handle) {
		pb_print(MSG_ERROR, "Invalid handle is specified(%d)\n", handle);
		return 0;
	}
	#endif /* PB_RING_DEBUG_PREC_CHECK */
	used_size = rings [handle].tail - rings [handle].head;
	if (used_size < 0L) {
		used_size += rings [handle].size;
	}
	return (used_size + size) < rings [handle].size;
}

int pb_ring_push (const int handle, const int size, const char* buf) {
	int write_head;
	int write_tail;
	int write_left = 0;
	struct pb_ring* ring;
	if (!pb_ring_is_enough (handle, size)) {
		pb_print(MSG_ERROR, "Ring No Space to write(%d)\n", size);
		return 0;
	}
	ring = &rings [handle];

	write_head = ring->tail;
	if (write_head == ring->size) {
		write_head = 0;
	}

	write_tail = write_head + size;
	if (write_tail > ring->size) {
		write_left = write_tail - ring->size;
		write_tail = ring->size;
	}

	memcpy (ring->buf + write_head, buf, write_tail - write_head);
	if (write_left) {
		memcpy (ring->buf, buf + write_tail - write_head, write_left);
		write_tail = write_left;
	}

	ring->tail = write_tail;
	if (ring->tail == ring->head) {
		pb_print(MSG_ERROR, "Ring Rounded\n");
	}
	return size;
}

int pb_ring_push_back (const int handle, const int size) {
	struct pb_ring* ring;

	if (pb_ring_is_enough (handle, size) == 0L) {
		return 0;
	}

	ring = &rings [handle];

	if (ring->head >= size) {
		ring->head -= size;
	} else {
		ring->head = ring->size + ring->head - size;
	}
	return size;
}

int pb_ring_pull (const int handle, const int size, char* buf) {
	struct pb_ring* ring;
	int pull_size = 0;
	int half_size = 0;
	#if defined  PB_RING_DEBUG_PREC_CHECK
	if (handle < 0 || PB_MAX_NUM_OF_INSTANCE <= handle) {
		pb_print(MSG_ERROR, "Invalid handle is specified(%d)\n", handle);
		return 0;
	}
	#endif /* PB_RING_DEBUG_PREC_CHECK */
	ring = &rings [handle];

	if (ring->head < ring->tail) {
		pull_size = ring->tail - ring->head;
		if (pull_size > size) {
			pull_size = size;
		}
		memcpy (buf, ring->buf + ring->head, pull_size);
		ring->head += pull_size;
		return pull_size;
	} else if (ring->head > ring->tail) {
		half_size = ring->size - ring->head;
		if (half_size > size) {
			half_size = size;
		}
		memcpy (buf, ring->buf + ring->head, half_size);
		if (half_size == size) {
			ring->head = (ring->head + half_size) % ring->size;
			return half_size;
		}
		pull_size = size - half_size;
		if (pull_size > ring->tail) {
			pull_size = ring->tail;
		}

		memcpy (buf + half_size, ring->buf, pull_size);
		ring->head = pull_size;
		return pull_size + half_size;
	} else {
		return 0;
	}
}

