/* security/kyocera/kclsm.c  (kclsm LSM module)
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2017 KYOCERA Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */
#include <linux/security.h>
#include <linux/path.h>
#include <linux/fs.h>
#include <linux/kclsm_sysfs.h>
#include "kclsm.h"

static int kclsm_enable = 0;
static int __init kclsm_setup(char *str)
{
        unsigned long enable;
        if (!kstrtoul(str, 0, &enable))
                kclsm_enable = enable ? 1 : 0;
        return 1;
}
__setup("kclsm=", kclsm_setup);

static int deny_uart2access = 0;
static int __init deny_uart2access_setup(char *str)
{
        unsigned long flag;
        if (!kstrtoul(str, 0, &flag))
                deny_uart2access = flag;
        return 1;
}
__setup("deny_uart2access=", deny_uart2access_setup);

int get_kclsm_enable(void)
{
	return kclsm_enable;
}
int get_deny_uart2access(void)
{
	return deny_uart2access;
}
int kclsm_restriction_check(struct path *path)
{
	char *ptr, *target_path =NULL;
	int ret = 0;
	pid_t gl_pid = 0;

	ptr = kmalloc(PATH_MAX, GFP_KERNEL);
	if (!ptr)
		return -ENOMEM;

	target_path = d_path(path, ptr, PATH_MAX);

	if (is_target_path(target_path) == false) {
		pr_debug("%s path is not target = %s\n", __func__, target_path);
		goto out;
	}

	/* allow process marked as dna_app */
	if (!current)
		goto out_err;
	if (find_stored_dna_app(current->pid) != NULL) {
		pr_debug("%s pid is dna_apps = %d\n", __func__, current->pid);
		goto out;
	}

	/* allow threads created from process marked as dna_app */
	if (!current->group_leader)
		goto out_err;
	gl_pid = current->group_leader->pid;
	if (find_stored_dna_app(gl_pid) != NULL) {
		pr_debug("%s gl_pid is dna_apps = %d\n", __func__, gl_pid);
		goto out;
	}

out_err:
	pr_warn("access restriction pid=%d target_path=%s\n",
		current->pid, target_path);
	ret = -EPERM;
out:
	kfree(ptr);
	return KCLSM_RETURN(ret);
}

static int kclsm_file_open(struct file *file, const struct cred *cred)
{
	int ret =0;
//	int i_mode = file->f_inode->i_mode;

	pr_debug("KCLSM: %s start\n",__func__);

	if ((ret = kclsm_restriction_check(&(file->f_path))) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_unlink(struct path *dir, struct dentry *dentry)
{
	int ret =0;
	struct path unlink_path;

	pr_debug("KCLSM: %s start\n",__func__);

	unlink_path.mnt = dir->mnt;
	unlink_path.dentry = dentry;

	if ((ret = kclsm_restriction_check(&unlink_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_mkdir(struct path *dir, struct dentry *dentry, umode_t mode)
{
	int ret =0;
	struct path mk_path;

	pr_debug("KCLSM: %s start\n",__func__);

	mk_path.mnt = dir->mnt;
	mk_path.dentry = dentry;

	if ((ret = kclsm_restriction_check(&mk_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_rmdir(struct path *dir, struct dentry *dentry)
{
	int ret =0;
	struct path rm_path;

	pr_debug("KCLSM: %s start\n",__func__);

	rm_path.mnt = dir->mnt;
	rm_path.dentry = dentry;

	if ((ret = kclsm_restriction_check(&rm_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_mknod(struct path *dir, struct dentry *dentry, umode_t mode, unsigned int dev)
{
	int ret =0;
	struct path mknod_path;

	pr_debug("KCLSM: %s start\n",__func__);

	mknod_path.mnt = dir->mnt;
	mknod_path.dentry = dentry;

	if ((ret = kclsm_restriction_check(&mknod_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_truncate(struct path *path)
{
	int ret =0;

	pr_debug("KCLSM: %s start\n",__func__);

	if ((ret = kclsm_restriction_check(path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_symlink(struct path *dir, struct dentry *dentry, const char *old_name)
{
	int ret =0;
	struct path slink_path;

	pr_debug("KCLSM: %s start\n",__func__);

	slink_path.mnt = dir->mnt;
	slink_path.dentry = dentry;

	if ((ret = kclsm_restriction_check(&slink_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_link(struct dentry *old_dentry, struct path *new_dir, struct dentry *new_dentry)
{
	int ret =0;
	struct path link_path;

	pr_debug("KCLSM: %s start\n",__func__);

	link_path.mnt = new_dir->mnt;
	link_path.dentry = new_dentry;

	if ((ret = kclsm_restriction_check(&link_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_rename(struct path *old_dir, struct dentry *old_dentry, struct path *new_dir, struct dentry *new_dentry)
{
	int ret =0;
	struct path old_path, new_path;

	pr_debug("KCLSM: %s start\n",__func__);

	old_path.mnt = old_dir->mnt;
	old_path.dentry = old_dentry;

	if ((ret = kclsm_restriction_check(&old_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
		goto out;
	}

	new_path.mnt = new_dir->mnt;
	new_path.dentry = new_dentry;

	if ((ret = kclsm_restriction_check(&new_path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

out:
	return KCLSM_RETURN(ret);
}

static int kclsm_path_chmod(struct path *path, umode_t mode)
{
	int ret =0;

	pr_debug("KCLSM: %s start\n",__func__);
	if ((ret = kclsm_restriction_check(path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}
	return KCLSM_RETURN(ret);
}

static int kclsm_path_chown(struct path *path, kuid_t uid, kgid_t gid)
{
	int ret =0;

	pr_debug("KCLSM: %s start\n",__func__);
	if ((ret = kclsm_restriction_check(path)) < 0) {
		pr_warn("no permission in %s pid=%d\n",
		__func__, current->pid);
	}

	return KCLSM_RETURN(ret);
}

static int kclsm_path_chroot(struct path *path)
{
	if (current->pid == 1)
		return 0;
	else
		return -EPERM;
}

static struct security_operations kclsm_security_operations = {
	.name = "kclsm",
	.file_open = kclsm_file_open,
	.path_unlink = kclsm_path_unlink,
	.path_mkdir  = kclsm_path_mkdir,
	.path_rmdir = kclsm_path_rmdir,
	.path_mknod  = kclsm_path_mknod,
	.path_truncate = kclsm_path_truncate,
	.path_symlink = kclsm_path_symlink,
	.path_link = kclsm_path_link,
	.path_rename = kclsm_path_rename,
	.path_chmod = kclsm_path_chmod,
	.path_chown = kclsm_path_chown,
	.path_chroot = kclsm_path_chroot,
};

static int __init kclsm_init(void)
{
	int ret;

	ret = register_security(&kclsm_security_operations);
	if (ret) {
		pr_err("Unable to register kclsm\n");
		return ret;
	}

	pr_info("kclsm initialized\n");

	return 0;
}
security_initcall(kclsm_init);
