/*
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2018 KYOCERA Corporation
 * (C) 2019 KYOCERA Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 and
 * only version 2 as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <linux/module.h>
#include <linux/spi/spi.h>
#include <linux/of_gpio.h>
#include <linux/delay.h>
#include <sound/soc.h>
#include "ak4633.h"												/* AUD_06_0242 */

static int						g_pdn_gpio	= 0;
static struct spi_device*		g_spi		= 0;
static int						g_invert	= 0;

#ifdef CONFIG_KYOCERA_MSND_NO_AUDIO
int ak4633_write( u8 in_addr, u8 in_data ){ return 0; }
int ak4633_read( u8 in_addr, u8* out_data ){ return 0; }
int ak4633_power( u8 enable ){ return 0; }
int ak4633_set_invert( void ){ return 0; }
#else /* CONFIG_KYOCERA_MSND_NO_AUDIO */
/*===========================================================================

FUNCTION : ak4633_write

===========================================================================*/
int ak4633_write(
	u8					in_addr,
	u8					in_data )
{
	struct spi_transfer	w_spi_transfer;
	struct spi_message	w_spi_message;
	unsigned short		w_tx_buf;
	int					w_rc;
/**/
	w_tx_buf = 0xa000;
	w_tx_buf |= (unsigned short)in_addr << 8;
	w_tx_buf |= in_data;

	if( g_invert ){
		w_tx_buf = ~w_tx_buf;
	}

	memset( &w_spi_transfer, 0, sizeof(w_spi_transfer) );
	w_spi_transfer.tx_buf = &w_tx_buf;
	w_spi_transfer.len = 2;

	spi_message_init( &w_spi_message );
	spi_message_add_tail( &w_spi_transfer, &w_spi_message );
	w_rc = spi_sync( g_spi, &w_spi_message );

	pr_info("%s w_rc=%x w_tx_buf=%x\n", __func__, (int)w_rc, (int)w_tx_buf);	/* AUD_06_0259 */

	return w_rc;
}

/*===========================================================================

FUNCTION : ak4633_read

===========================================================================*/
int ak4633_read(
	u8					in_addr,
	u8*					out_data )
{
	struct spi_transfer	w_spi_transfer;
	struct spi_message	w_spi_message;
	unsigned short		w_tx_buf;
	unsigned short		w_rx_buf;
	int					w_rc;
/**/
	w_tx_buf = 0x8000;
	w_tx_buf |= (unsigned short)in_addr << 8;
	w_tx_buf |= 0x00ff;

	if( g_invert ){
		w_tx_buf = ~w_tx_buf;
	}

	memset( &w_spi_transfer, 0, sizeof(w_spi_transfer) );
	w_spi_transfer.tx_buf = &w_tx_buf;
	w_spi_transfer.rx_buf = &w_rx_buf;
	w_spi_transfer.len = 2;

	spi_message_init( &w_spi_message );
	spi_message_add_tail( &w_spi_transfer, &w_spi_message );
	w_rc = spi_sync( g_spi, &w_spi_message );

	*out_data = (u8)w_rx_buf;

	pr_info("%s w_rc=%x w_tx_buf=%x w_rx_buf=%x\n", __func__, (int)w_rc, (int)w_tx_buf, (int)w_rx_buf );	/* AUD_06_0259 */
	return w_rc;
}

/* DA14A Start AUD_06_0289 */
/*===========================================================================

FUNCTION : ak4633_power

===========================================================================*/
int ak4633_power( u8 enable )
{
	int						ret;
	int						w_rc;
/**/
	ret = 0;

	pr_info( "ak4633_power %d\n", enable );
	w_rc = gpio_request( g_pdn_gpio, "ak4633 reset" );
	if( w_rc ){
		pr_err( "ak4633 gpio_request()=%x\n", w_rc );
		ret = (-1);
		return ret;
	}

	w_rc = gpio_direction_output( g_pdn_gpio, 1 );
	if( w_rc == 0 ){
		if( enable ){
			gpio_set_value( g_pdn_gpio, 0 );
			msleep(1);
			gpio_set_value( g_pdn_gpio, 1 );
		}else{
			gpio_set_value( g_pdn_gpio, 0 );
		}
	}else{
		pr_err( "gpio_direction_output()=%x\n", g_pdn_gpio );
		ret = (-2);
	}

	gpio_free( g_pdn_gpio );

	return ret;
}
/* DA14A End AUD_06_0289 */

/*===========================================================================

FUNCTION : ak4633_set_invert

===========================================================================*/
int ak4633_set_invert( void )
{
/* AUD_06_0243:s */
	int					ret;
	int					w_rc;
	u32					w_result;
/**/
	w_rc = ak4633_selfcheck( &w_result );
	if( ( w_rc != 0 ) || ( w_result != 0 ) ){
		pr_info("%s w_rc=%x w_result=0x%x g_invert %d->%d\n", __func__, (int)w_rc, w_result, g_invert, ~g_invert );	/* AUD_06_0259 */
		g_invert = ~g_invert;
	}

	w_rc = ak4633_selfcheck( &w_result );
	if( ( w_rc != 0 ) || ( w_result != 0 ) ){
		pr_err("%s w_rc=%x w_result=0x%x\n", __func__, (int)w_rc, w_result );
		w_rc = (-1);
	}
/* AUD_06_0243:e */
	return w_rc; /* AUD_06_0277 */
}
#endif /* CONFIG_KYOCERA_MSND_NO_AUDIO */

/*===========================================================================

FUNCTION : ak4633_reset

===========================================================================*/
int ak4633_reset( void )
{
	int						ret;
	int						w_rc;
/**/
	ret = 0;
	
	w_rc = gpio_request( g_pdn_gpio, "ak4633 reset" );
	if( w_rc ){
		pr_err( "ak4633 gpio_request()=%x\n", w_rc );
		ret = (-1);
		return ret;
	}

	w_rc = gpio_direction_output( g_pdn_gpio, 1 );
	if( w_rc == 0 ){
		gpio_set_value( g_pdn_gpio, 0 );
		msleep(1);
		gpio_set_value( g_pdn_gpio, 1 );
	}else{
		pr_err( "gpio_direction_output()=%x\n", g_pdn_gpio );
		ret = (-2);
	}

	gpio_free( g_pdn_gpio );

	return ret;
}

/* AUD_06_0242:s */
/*===========================================================================

FUNCTION : ak4633_write_bits

===========================================================================*/
int ak4633_write_bits(
	u8					in_addr,
	u8					in_data,
	u8					in_mask )
{
	u8					w_data;
	int					w_rc;
/**/
	w_rc = ak4633_read( in_addr, &w_data );

	in_data &= in_mask;
	w_data &= ~in_mask;
	in_data |= w_data;

	if( w_rc == 0 ){
		w_rc = ak4633_write( in_addr, in_data );
	}
	pr_info("%s w_rc=%x in_data=%x\n", __func__, (int)w_rc, (int)in_data);	/* AUD_06_0259 */

	return w_rc;
}
/* AUD_06_0242:e */
/* AUD_06_0243:s */
/*===========================================================================

FUNCTION : ak4633_selfcheck

===========================================================================*/
int ak4633_selfcheck(
	 u32*				out_result )
{
	int					ret;
	int					w_rc;
	u8					w_data;
/**/
	ret = 0;
	*out_result = 0;

	w_rc = ak4633_write_bits( 0x00, 0x20, 0x20 );
	w_rc |= ak4633_read( 0x00, &w_data );
	if( ( w_data & 0x22 ) != 0x20 ){
		*out_result = 1;
	}

	w_rc = ak4633_write_bits( 0x00, 0x00, 0x20 );
	w_rc |= ak4633_read( 0x00, &w_data );
	if( ( w_data & 0x22 ) != 0x00 ){
		*out_result = 1;
	}

	if( w_rc ){
		ret = (-1);
	}

	pr_debug( "ak4633_selfcheck %x ret=%x out_result=%x\n", __LINE__, ret , *out_result );
	return ret;
}
/* AUD_06_0243:e */
/*===========================================================================

FUNCTION : ak4633_spi_probe

===========================================================================*/
static int ak4633_spi_probe(
	struct spi_device*	in_spi )
{
	int						ret;
	int						w_rc;
	struct device_node*		w_np;
	int						w_gpio;
	enum of_gpio_flags		w_flags;
/**/
	pr_debug("%s\n", __func__);

	ret = 0;

	in_spi->bits_per_word = 16;
	w_rc = spi_setup( in_spi );
	if ( w_rc < 0 ){
		pr_err( "%s w_rc=%x\n", __func__, w_rc );
		ret = (-1);
		return ret;
	}

	w_np = in_spi->dev.of_node;
	if( !w_np ){
		pr_err( "%s w_np=%x\n", __func__, w_np ); /* AUD_06_0277 */
		ret = (-2);
		return ret;
	}

	w_gpio = of_get_named_gpio_flags( w_np, "pdn-gpio", 0, &w_flags );
	w_rc = gpio_is_valid( w_gpio );
	if( !w_rc ){
		pr_err( "%s w_rc=%x\n", __func__, w_rc ); /* AUD_06_0277 */
		ret = (-3);
		return ret;
	}

	pr_info( "ak4633 PDN gpio=%d\n", w_gpio );

	g_spi = in_spi;
	g_pdn_gpio = w_gpio;

/* DA14D Start AUD_06_0289 */
#if 0
	w_rc = ak4633_reset();
	if( w_rc ){ /* AUD_06_0277 */
		pr_err( "%s w_rc=%x\n", __func__, w_rc );
		ret = (-4);
		return ret;
	}
#endif
/* DA14D End AUD_06_0289 */

	pr_debug("%s w_rc=%x\n", __func__, w_rc);
	return ret;
}

/*===========================================================================

FUNCTION : ak4633_spi_remove

===========================================================================*/
static int ak4633_spi_remove(
	struct spi_device*	in_spi )
{
	pr_debug("%s\n", __func__);
	snd_soc_unregister_codec( &in_spi->dev );
	return 0;
}

/*===========================================================================
===========================================================================*/
static const struct of_device_id ak4633_of_match[] = {
	{ .compatible = "asahi-kasei,ak4633", },
	{ }
};
MODULE_DEVICE_TABLE(of, ak4633_of_match);

static const struct spi_device_id ak4633_id_table[] = {
	{ "ak4633", 0 },
	{ }
};
MODULE_DEVICE_TABLE(spi, ak4633_id_table);

static struct spi_driver ak4633_spi_driver = {
	.driver  = {
		.name   = "ak4633",
		.owner  = THIS_MODULE,
		.of_match_table = ak4633_of_match,
	},
	.id_table = ak4633_id_table,
	.probe  = ak4633_spi_probe,
	.remove = ak4633_spi_remove,
};

module_spi_driver(ak4633_spi_driver);

MODULE_AUTHOR("KYOCERA Corporation");
MODULE_DESCRIPTION("Asahi Kasei AK4633 ALSA SoC driver");
MODULE_LICENSE("GPL");

