 /***************************************************************************
 *
 * This software is contributed or developed by KYOCERA Corporation.
 * (C) 2019 KYOCERA Corporation
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 *****************************************************************************/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <linux/usb.h>
#include <linux/usb/usbnet.h>
#include <linux/slab.h>
#include "smsc95xx.h"

#define CONFIG_PARSER_SIZE     0x00000100UL
#define CONFIG_PARSE_FIRST     0x00000001L
#define CONFIG_PARSE_SKIP      0xFFFFFFFFL

#define CONFIG_TARGET_UNKNOWN  0x00000000L
#define CONFIG_TARGET_DATA     0x00000001L
#define CONFIG_TARGET_REG      0x00000002L

#define CONFIG_CHAR_NULL       0x00
#define CONFIG_CHAR_RETURN     0x0A
#define CONFIG_CHAR_COMMA      0x2C

#define CONFIG_OFFSET_BLOCK    0x00000001L
#define CONFIG_OFFSET_DATA     0x00000002L
#define CONFIG_NUM_REG         0x0000000EL
#define CONFIG_NUM_DATA        0x0000000AL

#define CONFIG_OFFSET_SUFFIX00 0x00000000L
#define CONFIG_OFFSET_SUFFIX01 0x00000001L
#define CONFIG_OFFSET_SUFFIX02 0x00000002L
#define CONFIG_OFFSET_SUFFIX03 0x00000003L
#define CONFIG_OFFSET_SUFFIX04 0x00000004L
#define CONFIG_OFFSET_SUFFIX05 0x00000005L
#define CONFIG_OFFSET_SUFFIX06 0x00000006L
#define CONFIG_OFFSET_SUFFIX07 0x00000007L
#define CONFIG_OFFSET_SUFFIX08 0x00000008L
#define CONFIG_OFFSET_SUFFIX09 0x00000009L
#define CONFIG_OFFSET_SUFFIX10 0x0000000AL
#define CONFIG_OFFSET_SUFFIX11 0x0000000BL
#define CONFIG_OFFSET_SUFFIX12 0x0000000CL
#define CONFIG_OFFSET_SUFFIX13 0x0000000DL

#define REGISTER_DEVICE_MAC    0x00000000UL
#define REGISTER_DEVICE_PHY_S  0x00000001UL
#define REGISTER_DEVICE_PHY_R  0x00000002UL

#define REGISTER_ACCESS_READ   0x00000000UL
#define REGISTER_ACCESS_WRITE  0x00000001UL

#define CONFIG_PARSER_OFF      0x00000000UL
#define CONFIG_PARSER_ON       0x00000001UL

#ifndef MAX
#define MAX(a,b) ((a)>(b)?(a):(b))
#endif /* MAX */

#ifndef MIN
#define MIN(a,b) ((a)<(b)?(a):(b))
#endif /* MIN */

#define FEATURE_8_WAKEUP_FILTERS    (0x01)
#define FEATURE_PHY_NLP_CROSSOVER   (0x02)
#define FEATURE_REMOTE_WAKEUP       (0x04)

const char CONFIG_STRING_COMMA[]   = { 0x2C, 0x00 };
const char CONFIG_STRING_MAGIC_R[] = { 0x52, 0x45, 0x47, 0x00 };
const char CONFIG_STRING_MAGIC_D[] = { 0x44, 0x41, 0x54, 0x41, 0x00 };
const char CONFIG_STRING_FNAME[]   = { 0x2F, 0x76, 0x65, 0x6E, 0x64, 0x6F, 0x72, 0x2F,
                                       0x61, 0x70, 0x70, 0x2F, 0x65, 0x74, 0x63, 0x2F,
                                       0x65, 0x74, 0x68, 0x65, 0x72, 0x6E, 0x65, 0x74,
                                       0x5F, 0x63, 0x6F, 0x6E, 0x66, 0x69, 0x67, 0x2E,
                                       0x63, 0x73, 0x76, 0x00 };

struct smsc95xx_priv {
	u32 mac_cr;
	u32 hash_hi;
	u32 hash_lo;
	u32 wolopts;
	spinlock_t mac_cr_lock;
	u8 features;
	u8 suspend_flags;
	struct mutex access_mutex;
	u32 mii_data;
	u32 mii_access;
};

struct smsc95xx_config {
	char block[CONFIG_PARSER_SIZE+1];
	u32  param[CONFIG_NUM_REG];
};

struct smsc95xx_reg_config {
	struct smsc95xx_config c;
	struct list_head whole;
	struct list_head part;
};

struct mii_infomation {
	u32 data[CONFIG_NUM_DATA];
	struct ethtool_cmd ecmd;
	int (* r_reg)(struct usbnet *dev, u32 index, u32 *data, int in_pm);
	int (* w_reg)(struct usbnet *dev, u32 index, u32  data, int in_pm);
};

static struct mii_infomation mii_info;
LIST_HEAD(smsc95xx_whole_list_head);
LIST_HEAD(smsc95xx_part_list_head);

/*****************************************************************************/
/* CONFIG FILE PARSER FUNCTIONS                                              */
/*****************************************************************************/

static int smsc95xx_cfile_parser_reg(struct smsc95xx_config *p, int item, char *str)
{
	int ret    = 0;
	int offset = 0;
	unsigned long val = 0;
	size_t str_length = 0;

	if (item == CONFIG_OFFSET_BLOCK) {
		str_length = MIN(CONFIG_PARSER_SIZE, strlen(str));
		memcpy(p->block, str, str_length);
	} else if ((0 < item) && ((item - CONFIG_OFFSET_DATA) < CONFIG_NUM_REG)) {
		offset = (item - CONFIG_OFFSET_DATA);
		ret = kstrtoul(str, 0, &val);
		if (ret == 0) {
			switch (offset) {
			case CONFIG_OFFSET_SUFFIX00:
			case CONFIG_OFFSET_SUFFIX01:
			case CONFIG_OFFSET_SUFFIX02:
			case CONFIG_OFFSET_SUFFIX03:
			case CONFIG_OFFSET_SUFFIX05:
			case CONFIG_OFFSET_SUFFIX06:
			case CONFIG_OFFSET_SUFFIX07:
				p->param[offset] = val;
				break;
			default:
				if ((val == CONFIG_PARSER_OFF) || (val == CONFIG_PARSER_ON)) {
					p->param[offset] = val;
				} else {
					pr_debug("%s: Syntax Error: Invalid value (offset=%d).\n", __func__, offset);
					ret = -EINVAL;
				}
				break;
			}
		}
	} else {
		pr_debug("%s: Syntax Error: Invalid offset.\n", __func__);
		ret = -EINVAL;
	}

	return ret;
}

static int smsc95xx_cfile_parser_data(struct usbnet *dev, int item, char *str)
{
	int ret    = 0;
	int offset = 0;
	unsigned long val = 0;

	if (item == CONFIG_OFFSET_BLOCK) {
		/* skip */
	} else if ((0 < item) && ((item - CONFIG_OFFSET_DATA) < CONFIG_NUM_DATA)) {
		offset = (item - CONFIG_OFFSET_DATA);
		ret = kstrtoul(str, 0, &val);
		if (ret == 0) {
			switch (offset) {
			case CONFIG_OFFSET_SUFFIX01:
			case CONFIG_OFFSET_SUFFIX02:
			case CONFIG_OFFSET_SUFFIX04:
			case CONFIG_OFFSET_SUFFIX06:
			case CONFIG_OFFSET_SUFFIX07:
			case CONFIG_OFFSET_SUFFIX08:
			case CONFIG_OFFSET_SUFFIX09:
				mii_info.data[offset] = val;
				break;
			default:
				if ((val == CONFIG_PARSER_OFF) || (val == CONFIG_PARSER_ON)) {
					mii_info.data[offset] = val;
				} else {
					pr_debug("%s: Syntax Error: Invalid value (offset=%d).\n", __func__, offset);
					ret = -EINVAL;
				}
				break;
			}
		}
	} else {
		pr_debug("%s: Syntax Error: Invalid offset.\n", __func__);
		ret = -EINVAL;
	}

	return ret;
}

static struct list_head* smsc95xx_cfile_get_list(const char *block)
{
	struct smsc95xx_reg_config *cdata = NULL;
	size_t len = 0;

	INIT_LIST_HEAD(&smsc95xx_part_list_head);

	list_for_each_entry(cdata, &smsc95xx_whole_list_head, whole) {
		len = MAX(strlen(&cdata->c.block[0]), strlen(block));
		if (0 == memcmp(&cdata->c.block, block, len)) {
			list_add_tail(&cdata->part, &smsc95xx_part_list_head);
		}
	}

	return (list_empty(&smsc95xx_part_list_head)) ? NULL : &smsc95xx_part_list_head;
}

void smsc95xx_cfile_exit (void)
{
	struct smsc95xx_reg_config *c = NULL;
	struct smsc95xx_reg_config *s = NULL;

	(void)memset(&mii_info, 0, sizeof(struct mii_infomation));

	if (list_empty(&smsc95xx_whole_list_head)) {
		pr_info("%s: whole list is empty.\n", __func__);
	} else {
		list_for_each_entry_safe(c, s, &smsc95xx_whole_list_head, whole) {
			list_del(&c->whole);
			kfree(c);
		}
	}
}

int smsc95xx_cfile_init (struct usbnet *dev,
	int (* r_reg)(struct usbnet *dev, u32 index, u32 *data, int in_pm),
	int (* w_reg)(struct usbnet *dev, u32 index, u32  data, int in_pm))
{
	int   ret      = 0;
	int   r_size   = 0;
	int   r_seek   = 0;
	int   offset   = 0;
	int   target   = 0;
	bool  d_parsed = false;
	bool  r_parsed = false;

	char *buff    = NULL;
	char *strp    = NULL;
	char *temp    = NULL;
	char *delimit = NULL;

	struct smsc95xx_priv       *pdata = NULL;
	struct smsc95xx_reg_config *cdata = NULL;
	struct file                *file  = NULL;

	(void)memset(&mii_info, 0, sizeof(struct mii_infomation));

	INIT_LIST_HEAD(&smsc95xx_whole_list_head);
	INIT_LIST_HEAD(&smsc95xx_part_list_head);

	if ((dev == NULL) || (r_reg == NULL) || (w_reg == NULL)) {
		pr_debug("%s: Invalid argument.\n", __func__);
		return -EINVAL;
	}

	buff = (char *)kzalloc(CONFIG_PARSER_SIZE + 1, GFP_KERNEL);
	if (buff == NULL) {
		pr_debug("%s: failed to allocate memory.\n", __func__);
		return -ENOMEM;
	}

	file = filp_open(CONFIG_STRING_FNAME, O_RDONLY, 0);
	if (IS_ERR(file)) {
		pr_debug("%s: failed to open file.\n", __func__);
		ret = -EIO;
		goto cleanup;
	}

	do {
		target = CONFIG_TARGET_UNKNOWN;
		offset = CONFIG_PARSE_FIRST;

		if ((r_parsed == false) && (cdata != NULL)) {
			list_del(&cdata->whole);
			kfree(cdata);
		}

		r_parsed = false;
		cdata    = NULL;

		strp = memset(buff, 0, CONFIG_PARSER_SIZE);
		r_size = kernel_read(file, r_seek, strp, CONFIG_PARSER_SIZE);
		if (r_size <= 0) {
			break;
		}

		temp = strchr(strp, CONFIG_CHAR_RETURN);
		if (temp != NULL) {
			*temp = CONFIG_CHAR_NULL;
			r_seek += ((temp + 1) - strp);
		} else {
			temp = strchr(strp, CONFIG_CHAR_COMMA);
			if (temp != NULL) {
				r_seek += r_size;
			} else {
				pr_debug("%s: Syntax Error: Length over.\n", __func__);
				ret = -EINVAL;
				break;
			}
		}

		do {
			delimit = strsep(&strp, CONFIG_STRING_COMMA);
			if ((delimit != NULL) && (delimit[0] != CONFIG_CHAR_NULL)) {

				if (strncmp(delimit, CONFIG_STRING_MAGIC_D, CONFIG_PARSER_SIZE) == 0) {
					if (d_parsed == false) {
						target = CONFIG_TARGET_DATA;
					} else {
						pr_debug("%s: Syntax Error: multiple definition of DATA.\n", __func__);
						ret = -EINVAL;
						goto cleanup;
					}
					continue;
				}

				if (strncmp(delimit, CONFIG_STRING_MAGIC_R, CONFIG_PARSER_SIZE) == 0) {
					if (r_parsed == false) {
						if (cdata == NULL) {
							cdata = (struct smsc95xx_reg_config *)kzalloc(sizeof(struct smsc95xx_reg_config), GFP_KERNEL);
							if (cdata != NULL) {
								list_add_tail(&cdata->whole, &smsc95xx_whole_list_head);
								target = CONFIG_TARGET_REG;
							} else {
								pr_debug("%s: Syntax Error: failed to allocate memory.\n", __func__);
								target = CONFIG_TARGET_UNKNOWN;
								offset = CONFIG_PARSE_SKIP;
							}
						}
					} else {
						pr_debug("%s: Syntax Error: multiple definition of REG.\n", __func__);
						ret    = 0;
						target = CONFIG_TARGET_UNKNOWN;
						offset = CONFIG_PARSE_SKIP;
						if (cdata != NULL) {
							list_del(&cdata->whole);
							kfree(cdata);
						}
						cdata = NULL;
					}
					continue;
				}

				if (offset == CONFIG_PARSE_SKIP) {
					pr_debug("%s: Syntax Error: Parse skip (fp:%d, target:%d, offset:%d)\n", __func__, r_seek, target, offset);
					continue;
				}

				if (target == CONFIG_TARGET_DATA) {
					ret = smsc95xx_cfile_parser_data(dev, offset, delimit);
					if (ret == 0) {
						if (offset == (CONFIG_NUM_DATA + CONFIG_OFFSET_BLOCK)) {
							d_parsed = true;
						}
						offset++;
					} else {
						pr_debug("%s: Syntax Error: SETTINGS (DATA(ret=%d))\n", __func__, ret);
						ret = -EINVAL;
						goto cleanup;
					}
					continue;
				}

				if (target == CONFIG_TARGET_REG) {
					ret = smsc95xx_cfile_parser_reg(&cdata->c, offset, delimit);
					if (ret == 0) {
						if (offset == (CONFIG_NUM_REG  + CONFIG_OFFSET_BLOCK)) {
							r_parsed = true;
						}
						offset++;
					} else {
						pr_debug("%s: Syntax Error: SETTINGS (REG (ret=%d))\n", __func__, ret);
						ret    = 0;
						target = CONFIG_TARGET_UNKNOWN;
						offset = CONFIG_PARSE_SKIP;
						list_del(&cdata->whole);
						kfree(cdata);
						cdata = NULL;
					}
					continue;
				}
			}
		} while (delimit != NULL);
	} while (0 < r_size);

	if (ret == 0) {
		if (d_parsed == false) {
			pr_debug("%s: Syntax Error: Not found DATA block.\n", __func__);
			ret = -EINVAL;
		} else {
			dev->mii.supports_gmii = 0;
			dev->mii.phy_id        = mii_info.data[CONFIG_OFFSET_SUFFIX00];
			dev->mii.phy_id_mask   = mii_info.data[CONFIG_OFFSET_SUFFIX01];
			dev->mii.reg_num_mask  = mii_info.data[CONFIG_OFFSET_SUFFIX02];
			mii_info.ecmd.autoneg  = mii_info.data[CONFIG_OFFSET_SUFFIX03];
			mii_info.ecmd.speed    = mii_info.data[CONFIG_OFFSET_SUFFIX04];
			mii_info.ecmd.duplex   = mii_info.data[CONFIG_OFFSET_SUFFIX05];

			pdata = (struct smsc95xx_priv *)(dev->data[0]);
			pdata->mii_access      = mii_info.data[CONFIG_OFFSET_SUFFIX06];
			pdata->mii_data        = mii_info.data[CONFIG_OFFSET_SUFFIX07];

			dev->mii.full_duplex = (mii_info.ecmd.duplex)  ? DUPLEX_FULL : DUPLEX_HALF;
			dev->mii.advertising = (mii_info.ecmd.autoneg) ? LPA_LPACK : 0;
			if (mii_info.ecmd.speed == SPEED_100) {
				if (mii_info.ecmd.duplex) {
					dev->mii.advertising = ADVERTISE_100FULL;
				} else {
					dev->mii.advertising = ADVERTISE_100HALF;
				}
			} else {
				if (mii_info.ecmd.duplex) {
					dev->mii.advertising = ADVERTISE_10FULL;
				} else {
					dev->mii.advertising = ADVERTISE_10HALF;
				}
			}
			mii_info.ecmd.advertising  = mii_lpa_to_ethtool_lpa_t(dev->mii.advertising);
			mii_info.ecmd.transceiver  = (dev->mii.phy_id) ? XCVR_EXTERNAL : XCVR_INTERNAL;
			mii_info.r_reg             = r_reg;
			mii_info.w_reg             = w_reg;
		}
	}

cleanup:
	if (!IS_ERR(file)) {
		(void)filp_close(file, NULL);
	}

	if (buff != NULL) {
		kfree(buff);
	}

	if (ret != 0) {
		smsc95xx_cfile_exit();
	}

	pr_debug("%s: finish ret=%d", __func__, ret);

	return ret;
}

/*****************************************************************************/
/* REGISTER ACCESS FUNCTIONS                                                 */
/*****************************************************************************/

int smsc95xx_cfile_register_control (struct usbnet *dev, const char *block, u32 *outparam)
{
	int ret        = 0;
	u32 r_val      = 0;
	u32 w_val      = 0;
	u32 addr_lo    = 0;
	u32 addr_hi    = 0;
	u32 *r_val_ptr = NULL;
	unsigned long flags = 0;

	struct list_head           *head  = NULL;
	struct smsc95xx_reg_config *cdata = NULL;
	struct smsc95xx_priv       *pdata = NULL;

	if ((dev == NULL) || (block == NULL)) {
		pr_debug("%s: Invalid argument.\n", __func__);
		return -EINVAL;
	}

	if ((mii_info.r_reg == NULL) || (mii_info.w_reg == NULL)) {
		pr_debug("%s: device not ready.\n", __func__);
		return -ENODEV;
	}

	pr_debug("%s: request block=%s\n", __func__, block);

	pdata   = (struct smsc95xx_priv *)(dev->data[0]);
	addr_lo = dev->net->dev_addr[0] | dev->net->dev_addr[1] << 8 | dev->net->dev_addr[2] << 16 | dev->net->dev_addr[3] << 24;
	addr_hi = dev->net->dev_addr[4] | dev->net->dev_addr[5] << 8;

	mutex_lock(&pdata->access_mutex);

	head = smsc95xx_cfile_get_list(block);
	if (head == NULL) {
		mutex_unlock(&pdata->access_mutex);
		pr_debug("%s: Setting of register is not necessary.\n", __func__);
		return 0;
	}

	list_for_each_entry(cdata, head, part) {
		if (cdata->c.param[CONFIG_OFFSET_SUFFIX04] == REGISTER_ACCESS_READ) {

			pr_debug("%s: REGISTER_ACCESS_READ\n", __func__);

			r_val_ptr = &r_val;
			if (cdata->c.param[CONFIG_OFFSET_SUFFIX10]) {
				r_val_ptr = &pdata->mac_cr;
			}

			if (cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_MAC) {
				ret = mii_info.r_reg(dev, cdata->c.param[CONFIG_OFFSET_SUFFIX01], r_val_ptr, 0);
			} else if (cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_S) {
				ret = dev->mii.mdio_read(dev->net, dev->mii.phy_id, cdata->c.param[CONFIG_OFFSET_SUFFIX02]);
				if (ret >= 0) {
					*r_val_ptr = ret;
					ret = 0;
				}
			} else if (cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_R) {
				ret = dev->mii.mdio_write(dev->net, dev->mii.phy_id, mii_info.data[CONFIG_OFFSET_SUFFIX08], cdata->c.param[CONFIG_OFFSET_SUFFIX03]);
				if (ret >= 0) {
					ret = dev->mii.mdio_read(dev->net, dev->mii.phy_id, mii_info.data[CONFIG_OFFSET_SUFFIX09]);
				}
				if (ret >= 0) {
					*r_val_ptr = ret;
					ret = 0;
				}
			} else {
				pr_debug("%s: invalid device type.\n", __func__, ret);
				continue;
			}

			if (ret < 0) {
				pr_debug("%s: Failed to register read. ret:%d\n", __func__, ret);
				break;
			}

			*r_val_ptr = (*r_val_ptr & cdata->c.param[CONFIG_OFFSET_SUFFIX07]);
			if (cdata->c.param[CONFIG_OFFSET_SUFFIX09]) {
				*r_val_ptr >>= 16;
				if ((*r_val_ptr == ID_REV_CHIP_ID_9500A_) || (*r_val_ptr == ID_REV_CHIP_ID_9530_) ||
				    (*r_val_ptr == ID_REV_CHIP_ID_89530_) || (*r_val_ptr == ID_REV_CHIP_ID_9730_)) {
					pdata->features = (FEATURE_8_WAKEUP_FILTERS | FEATURE_PHY_NLP_CROSSOVER | FEATURE_REMOTE_WAKEUP);
				} else if (*r_val_ptr == ID_REV_CHIP_ID_9512_) {
					pdata->features = FEATURE_8_WAKEUP_FILTERS;
				} else {
					/* do nothing */
				}
			}
			if ((cdata->c.param[CONFIG_OFFSET_SUFFIX13]) && (outparam != NULL)) {
				*outparam = *r_val_ptr;
			}

			pr_debug("%s: %s Register Read: id=%08x, val=%08x, ret=%d\n",
				__func__,
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_MAC)   ? "MAC":
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_S) ? "PHY-S" : "PHY-R",
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_MAC)   ? cdata->c.param[CONFIG_OFFSET_SUFFIX01]:
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_S) ? cdata->c.param[CONFIG_OFFSET_SUFFIX02] : cdata->c.param[CONFIG_OFFSET_SUFFIX03],
				*r_val_ptr, ret);
			continue;
		}

		if (cdata->c.param[CONFIG_OFFSET_SUFFIX04] == REGISTER_ACCESS_WRITE) {

			pr_debug("%s: REGISTER_ACCESS_WRITE\n", __func__);

			w_val = 0;
			if ((cdata->c.param[CONFIG_OFFSET_SUFFIX08]) && (r_val_ptr != NULL)) {
				w_val = *r_val_ptr;
			}

			if (cdata->c.param[CONFIG_OFFSET_SUFFIX10]) {
				spin_lock_irqsave(&pdata->mac_cr_lock, flags);
				w_val = (((w_val | pdata->mac_cr) & cdata->c.param[CONFIG_OFFSET_SUFFIX06]) | cdata->c.param[CONFIG_OFFSET_SUFFIX05]);
				pdata->mac_cr = w_val;
				spin_unlock_irqrestore(&pdata->mac_cr_lock, flags);
			} else if (cdata->c.param[CONFIG_OFFSET_SUFFIX11]) {
				w_val = addr_hi;
			} else if (cdata->c.param[CONFIG_OFFSET_SUFFIX12]) {
				w_val = addr_lo;
			} else {
				w_val = ((w_val & cdata->c.param[CONFIG_OFFSET_SUFFIX06]) | cdata->c.param[CONFIG_OFFSET_SUFFIX05]);
			}

			if (cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_MAC) {
				ret = mii_info.w_reg(dev, cdata->c.param[CONFIG_OFFSET_SUFFIX01], w_val, 0);
			} else if (cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_S) {
				ret = dev->mii.mdio_write(dev->net, dev->mii.phy_id, cdata->c.param[CONFIG_OFFSET_SUFFIX02], w_val);
			} else if (cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_R) {
				ret = dev->mii.mdio_write(dev->net, dev->mii.phy_id, mii_info.data[CONFIG_OFFSET_SUFFIX08], cdata->c.param[CONFIG_OFFSET_SUFFIX03]);
				if (ret >= 0) {
					ret = dev->mii.mdio_write(dev->net, dev->mii.phy_id, mii_info.data[CONFIG_OFFSET_SUFFIX09], w_val);
				}
			} else {
				pr_debug("%s: invalid device type.\n", __func__);
				continue;
			}
			if (ret < 0) {
				pr_debug("%s: Failed to register write. ret:%d\n", __func__, ret);
				break;
			}

			pr_debug("%s: %s Register Write: id=%08x, val=%08x, ret=%d\n",
				__func__,
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_MAC)   ? "MAC":
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_S) ? "PHY-S" : "PHY-R",
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_MAC)   ? cdata->c.param[CONFIG_OFFSET_SUFFIX01]:
				(cdata->c.param[CONFIG_OFFSET_SUFFIX00] == REGISTER_DEVICE_PHY_S) ? cdata->c.param[CONFIG_OFFSET_SUFFIX02] : cdata->c.param[CONFIG_OFFSET_SUFFIX03],
				w_val, ret);
			continue;
		}
	}

	mutex_unlock(&pdata->access_mutex);
	pr_debug("%s: finish ret=%d\n", __func__, ret);

	return ret;
}

void smsc95xx_cfile_parser_getdata (struct ethtool_cmd *ecmd)
{
	ecmd->advertising |= mii_info.ecmd.advertising;
	ecmd->supported    = ecmd->advertising;
	ecmd->transceiver  = mii_info.ecmd.transceiver;
	ecmd->autoneg      = mii_info.ecmd.autoneg;
	ecmd->speed        = mii_info.ecmd.speed;
	ecmd->duplex       = mii_info.ecmd.duplex;
}

EXPORT_SYMBOL(smsc95xx_cfile_exit);
EXPORT_SYMBOL(smsc95xx_cfile_init);
EXPORT_SYMBOL(smsc95xx_cfile_register_control);
EXPORT_SYMBOL(smsc95xx_cfile_parser_getdata);
